#include "ax_main.h"
#include "ax88179_178a.h"
#include "ax88279_179a_772d.h"

static int bEEE = 0;
module_param(bEEE, int, 0);
MODULE_PARM_DESC(bEEE, "EEE advertisement configuration");

static int bctrl = -1;
module_param(bctrl, int, 0);
MODULE_PARM_DESC(bctrl, "RX Bulk Control");

static int blwt = -1;
module_param(blwt, int, 0);
MODULE_PARM_DESC(blwt, "RX Bulk Timer Low");

static int bhit = -1;
module_param(bhit, int, 0);
MODULE_PARM_DESC(bhit, "RX Bulk Timer High");

static int bsize = -1;
module_param(bsize, int, 0);
MODULE_PARM_DESC(bsize, "RX Bulk Queue Size");

static int bifg = -1;
module_param(bifg, int, 0);
MODULE_PARM_DESC(bifg, "RX Bulk Inter Frame Gap");

static int
ax_submit_rx(struct ax_device *dev, struct rx_desc *desc, gfp_t mem_flags);

void ax_get_drvinfo(struct net_device *net, struct ethtool_drvinfo *info)
{
	struct ax_device *axdev = netdev_priv(net);

	strlcpy (info->driver, MODULENAME, sizeof(info->driver));
	strlcpy (info->version, DRIVER_VERSION, sizeof info->version);
	usb_make_path (axdev->udev, info->bus_info, sizeof info->bus_info);

	sprintf(info->fw_version, "v%d.%d.%d",
		axdev->fw_version[0],
		axdev->fw_version[1],
		axdev->fw_version[2]);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 10, 0)
int ax_get_settings(struct net_device *net, struct ethtool_cmd *cmd)
{
	struct ax_device *axdev = netdev_priv(net);
	return mii_ethtool_gset(&axdev->mii, cmd);
}

int ax_set_settings(struct net_device *net, struct ethtool_cmd *cmd)
{
	struct ax_device *axdev = netdev_priv(net);
	return mii_ethtool_sset(&axdev->mii, cmd);
}
#else
int ax_get_link_ksettings(struct net_device *netdev,
			  struct ethtool_link_ksettings *cmd)
{
	struct ax_device *axdev = netdev_priv(netdev);
	int ret;

	if (!axdev->mii.mdio_read)
		return -EOPNOTSUPP;

	ret = usb_autopm_get_interface(axdev->intf);
	if (ret < 0)
		return ret;

	mutex_lock(&axdev->control);

	mii_ethtool_get_link_ksettings(&axdev->mii, cmd);

	mutex_unlock(&axdev->control);

	usb_autopm_put_interface(axdev->intf);

	return 0;
}

int ax_set_link_ksettings(struct net_device *netdev,
			  const struct ethtool_link_ksettings *cmd)
{
	struct ax_device *axdev = netdev_priv(netdev);
	int ret;

	ret = usb_autopm_get_interface(axdev->intf);
	if (ret < 0)
		return ret;

	mutex_lock(&axdev->control);

	mii_ethtool_set_link_ksettings(&axdev->mii, cmd);

	mutex_unlock(&axdev->control);

	usb_autopm_put_interface(axdev->intf);

	return 0;
}
#endif
u32 ax_get_msglevel(struct net_device *netdev)
{
	struct ax_device *axdev = netdev_priv(netdev);

	return axdev->msg_enable;
}

void ax_set_msglevel(struct net_device *netdev, u32 value)
{
	struct ax_device *axdev = netdev_priv(netdev);

	axdev->msg_enable = value;
}

void ax_get_wol(struct net_device *net, struct ethtool_wolinfo *wolinfo)
{
	struct ax_device *axdev = netdev_priv(net);
	u8 reg8;
	int ret;

	ret = ax_read_cmd(axdev, AX_ACCESS_MAC, AX_MONITOR_MODE,
			  1, 1, &reg8, 0);
	if (ret < 0) {
		wolinfo->supported = 0;
		wolinfo->wolopts = 0;
		return;
	}

	wolinfo->supported = WAKE_PHY | WAKE_MAGIC;

	if (reg8 & AX_MONITOR_MODE_RWLC)
		wolinfo->wolopts |= WAKE_PHY;
	if (reg8 & AX_MONITOR_MODE_RWMP)
		wolinfo->wolopts |= WAKE_MAGIC;
}

int ax_set_wol(struct net_device *net, struct ethtool_wolinfo *wolinfo)
{
	struct ax_device *axdev = netdev_priv(net);
	u8 reg8 = 0;
	int ret;

	if (wolinfo->wolopts & WAKE_PHY)
		reg8 |= AX_MONITOR_MODE_RWLC;
	else
		reg8 &= ~AX_MONITOR_MODE_RWLC;

	if (wolinfo->wolopts & WAKE_MAGIC)
		reg8 |= AX_MONITOR_MODE_RWMP;
	else
		reg8 &= ~AX_MONITOR_MODE_RWMP;

	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_MONITOR_MODE, 1, 1, &reg8);
	if (ret < 0)
		return ret;

	return 0;
}


static int __ax_usb_read_cmd(struct ax_device *axdev, u8 cmd, u8 reqtype,
			     u16 value, u16 index, void *data, u16 size)
{
	void *buf = NULL;
	int err = -ENOMEM;

	netdev_dbg(axdev->netdev, "ax_usb_read_cmd cmd=0x%02x reqtype=%02x"
		   " value=0x%04x index=0x%04x size=%d\n",
		   cmd, reqtype, value, index, size);

	if (size) {
		buf = kmalloc(size, GFP_KERNEL);
		if (!buf)
			goto out;
	}

	err = usb_control_msg(axdev->udev, usb_rcvctrlpipe(axdev->udev, 0),
			      cmd, reqtype, value, index, buf, size,
			      USB_CTRL_GET_TIMEOUT);
	if (err > 0 && err <= size) {
		if (data)
			memcpy(data, buf, err);
		else
			netdev_dbg(axdev->netdev,
				   "Huh? Data requested but thrown away.\n");
	}
	kfree(buf);
out:
	return err;
}

static int __ax_usb_write_cmd(struct ax_device *axdev, u8 cmd, u8 reqtype,
			      u16 value, u16 index, const void *data, u16 size)
{
	void *buf = NULL;
	int err = -ENOMEM;

	netdev_dbg(axdev->netdev, "ax_usb_write_cmd cmd=0x%02x reqtype=%02x"
		   " value=0x%04x index=0x%04x size=%d\n",
		   cmd, reqtype, value, index, size);

	if (data) {
		buf = kmemdup(data, size, GFP_KERNEL);
		if (!buf)
			goto out;
	} else {
		if (size) {
			WARN_ON_ONCE(1);
			err = -EINVAL;
			goto out;
		}
	}

	err = usb_control_msg(axdev->udev, usb_sndctrlpipe(axdev->udev, 0),
			      cmd, reqtype, value, index, buf, size,
			      USB_CTRL_SET_TIMEOUT);
	kfree(buf);

out:
	return err;
}

static int __ax_read_cmd(struct ax_device *dev, u8 cmd, u8 reqtype, u16 value,
			 u16 index, void *data, u16 size)
{
	int ret;

	if (usb_autopm_get_interface(dev->intf) < 0)
		return -ENODEV;

	ret = __ax_usb_read_cmd(dev, cmd, reqtype, value, index,
				data, size);

	usb_autopm_put_interface(dev->intf);

	return ret;
}

static int __ax_write_cmd(struct ax_device *dev, u8 cmd, u8 reqtype, u16 value,
			  u16 index, const void *data, u16 size)
{
	int ret;

	if (usb_autopm_get_interface(dev->intf) < 0)
		return -ENODEV;

	ret = __ax_usb_write_cmd(dev, cmd, reqtype, value, index,
				 data, size);

	usb_autopm_put_interface(dev->intf);

	return ret;
}

static int __ax_read_cmd_nopm(struct ax_device *dev, u8 cmd, u8 reqtype,
			      u16 value, u16 index, void *data, u16 size)
{
	return __ax_usb_read_cmd(dev, cmd, reqtype, value, index,
				 data, size);
}

static int __ax_write_cmd_nopm(struct ax_device *dev, u8 cmd, u8 reqtype,
			       u16 value, u16 index, const void *data,
			       u16 size)
{
	return __ax_usb_write_cmd(dev, cmd, reqtype, value, index,
				  data, size);
}

static int __asix_read_cmd(struct ax_device *dev, u8 cmd, u16 value, u16 index,
			   u16 size, void *data, int in_pm)
{
	int ret;
	int (*fn)(struct ax_device *, u8, u8, u16, u16, void *, u16);

	if (!in_pm)
		fn = __ax_read_cmd;
	else
		fn = __ax_read_cmd_nopm;

	ret = fn(dev, cmd, USB_DIR_IN | USB_TYPE_VENDOR |
		 USB_RECIP_DEVICE, value, index, data, size);

	if (unlikely(ret < 0))
		netdev_warn(dev->netdev,
			    "Failed to read reg cmd 0x%04x, value 0x%04x,"
			    " index: 0x%x, size: %d (%d)\n",
			    cmd, value, index, size, ret);
	return ret;
}

static int __asix_write_cmd(struct ax_device *dev, u8 cmd, u16 value, u16 index,
			    u16 size, void *data, int in_pm)
{
	int ret;
	int (*fn)(struct ax_device *, u8, u8, u16, u16, const void *, u16);

	if (!in_pm)
		fn = __ax_write_cmd;
	else
		fn = __ax_write_cmd_nopm;

	ret = fn(dev, cmd, USB_DIR_OUT | USB_TYPE_VENDOR |
		 USB_RECIP_DEVICE, value, index, data, size);

	if (unlikely(ret < 0))
		netdev_warn(dev->netdev,
			    "Failed to write reg cmd 0x%04x, value 0x%04x,"
			    " index: 0x%x, size: %d (%d)\n",
			    cmd, value, index, size, ret);

	return ret;
}

int ax_read_cmd_nopm(struct ax_device *dev, u8 cmd, u16 value,
		     u16 index, u16 size, void *data, int eflag)
{
	int ret;

	if (eflag && (2 == size)) {
		u16 buf = 0;
		ret = __asix_read_cmd(dev, cmd, value, index, size, &buf, 1);
		le16_to_cpus(&buf);
		*((u16 *)data) = buf;
	} else if (eflag && (4 == size)) {
		u32 buf = 0;
		ret = __asix_read_cmd(dev, cmd, value, index, size, &buf, 1);
		le32_to_cpus(&buf);
		*((u32 *)data) = buf;
	} else {
		ret = __asix_read_cmd(dev, cmd, value, index, size, data, 1);
	}

	return ret;
}

int ax_write_cmd_nopm(struct ax_device *dev, u8 cmd, u16 value,
		      u16 index, u16 size, void *data)
{
	int ret;

	if (2 == size) {
		u16 buf = 0;
		buf = *((u16 *)data);
		cpu_to_le16s(&buf);
		ret = __asix_write_cmd(dev, cmd, value, index,
					  size, &buf, 1);
	} else {
		ret = __asix_write_cmd(dev, cmd, value, index,
					  size, data, 1);
	}

	return ret;
}

int ax_read_cmd(struct ax_device *dev, u8 cmd, u16 value, u16 index, u16 size,
		void *data, int eflag)
{

	int ret;

	if (eflag && (2 == size)) {
		u16 buf = 0;
		ret = __asix_read_cmd(dev, cmd, value, index, size, &buf, 0);
		le16_to_cpus(&buf);
		*((u16 *)data) = buf;
	} else if (eflag && (4 == size)) {
		u32 buf = 0;
		ret = __asix_read_cmd(dev, cmd, value, index, size, &buf, 0);
		le32_to_cpus(&buf);
		*((u32 *)data) = buf;
	} else {
		ret = __asix_read_cmd(dev, cmd, value, index, size, data, 0);
	}

	return ret;
}

int ax_write_cmd(struct ax_device *dev, u8 cmd, u16 value, u16 index, u16 size,
		 void *data)
{
	int ret;

	if (2 == size) {
		u16 buf = 0;
		buf = *((u16 *)data);
		cpu_to_le16s(&buf);
		ret = __asix_write_cmd(dev, cmd, value, index,
					size, &buf, 0);
	} else {
		ret = __asix_write_cmd(dev, cmd, value, index,
					size, data, 0);
	}

	return ret;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 20)
static void ax_async_write_callback(struct urb *urb, struct pt_regs *regs)
#else
static void ax_async_write_callback(struct urb *urb)
#endif
{
	struct ax_device_async_handle *asyncdata =
				(struct ax_device_async_handle *)urb->context;

	if (urb->status < 0)
		printk(KERN_ERR "ax88179_async_cmd_callback() failed with %d",
		       urb->status);

	kfree(asyncdata->req);
	kfree(asyncdata);
	usb_free_urb(urb);
}

int ax_write_cmd_async(struct ax_device *dev, u8 cmd, u16 value, u16 index,
		       u16 size, void *data)
{
	struct usb_ctrlrequest *req = NULL;
	int status = 0;
	struct urb *urb = NULL;
	void *buf = NULL;
	struct ax_device_async_handle *asyncdata = NULL;

	urb = usb_alloc_urb(0, GFP_ATOMIC);
	if (urb == NULL) {
		netdev_err(dev->netdev,
			   "Error allocating URB in write_cmd_async!");
		return -ENOMEM;
	}

	req = kmalloc(sizeof(struct usb_ctrlrequest), GFP_ATOMIC);
	if (req == NULL) {
		netdev_err(dev->netdev,
			   "Failed to allocate memory for control request");
		usb_free_urb(urb);
		return -ENOMEM;
	}

	asyncdata = (struct ax_device_async_handle*)
			kmalloc(sizeof(struct ax_device_async_handle),
				GFP_ATOMIC);
	if (asyncdata == NULL) {
		netdev_err(dev->netdev,
			   "Failed to allocate memory for async data");
		kfree(req);
		usb_free_urb(urb);
		return -ENOMEM;
	}

	asyncdata->req = req;
	
	if (size == 2) {
		asyncdata->rxctl = *((u16 *)data);
		cpu_to_le16s(&asyncdata->rxctl);
		buf = &asyncdata->rxctl;
	} else {
		memcpy(asyncdata->m_filter, data, size);
		buf = asyncdata->m_filter;
	}

	req->bRequestType = USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE;
	req->bRequest = cmd;
	req->wValue = cpu_to_le16(value);
	req->wIndex = cpu_to_le16(index);
	req->wLength = cpu_to_le16(size);

	usb_fill_control_urb(urb, dev->udev,
			     usb_sndctrlpipe(dev->udev, 0),
			     (void *)req, buf, size,
			     ax_async_write_callback, asyncdata);

	status = usb_submit_urb(urb, GFP_ATOMIC);
	if (status < 0) {
		netdev_err(dev->netdev,
			   "Error submitting the control message: status=%d",
			   status);
		kfree(req);
		kfree(asyncdata);
		usb_free_urb(urb);
	}

	return status;
}

int ax_mdio_read(struct net_device *netdev, int phy_id, int reg)
{
	struct ax_device *dev = netdev_priv(netdev);
	u16 res;

	ax_read_cmd(dev, AX_ACCESS_PHY, phy_id, (__u16)reg, 2, &res, 1);

	return res;
}

void ax_mdio_write(struct net_device *netdev, int phy_id, int reg, int val)
{
	struct ax_device *dev = netdev_priv(netdev);
	u16 res = (u16)val;

	ax_write_cmd(dev, AX_ACCESS_PHY, phy_id, (__u16)reg, 2, &res);
}

inline struct net_device_stats *ax_get_stats(struct net_device *dev)
{
	return &dev->stats;
}

static void ax_set_unplug(struct ax_device *axdev)
{
	if (axdev->udev->state == USB_STATE_NOTATTACHED) {
		set_bit(AX_UNPLUG, &axdev->flags);
		smp_mb__after_atomic();
	}
}

static int ax_check_tx_queue_not_empty(struct ax_device *axdev)
{
	if (!skb_queue_empty(&axdev->tx_queue)) return 0;

	return -1;
}

static bool ax_check_tx_queue_len(struct ax_device *axdev)
{
	if (skb_queue_len(&axdev->tx_queue) > axdev->tx_qlen) return true;

	return false;
}

static void ax_read_bulk_callback(struct urb *urb)
{
	struct net_device *netdev;
	int status = urb->status;
	struct rx_desc *desc;
	struct ax_device *axdev;

	desc = urb->context;
	if (!desc)
		return;

	axdev = desc->context;
	if (!axdev)
		return;

	if (test_bit(AX_UNPLUG, &axdev->flags))
		return;
	if (!test_bit(AX_ENABLE, &axdev->flags))
		return;

	netdev = axdev->netdev;

	if (!netif_carrier_ok(netdev))
		return;

	usb_mark_last_busy(axdev->udev);

	if (status < 0)
		printk("RX callback: %d", status);

	switch (status) {
	case 0:
		if (urb->actual_length < ETH_ZLEN)
			break;

		spin_lock(&axdev->rx_lock);
		list_add_tail(&desc->list, &axdev->rx_done);
		spin_unlock(&axdev->rx_lock);
		napi_schedule(&axdev->napi);
		return;
	case -ESHUTDOWN:
		ax_set_unplug(axdev);
		netif_device_detach(axdev->netdev);
		return;
	case -ENOENT:
		return;
	case -ETIME:
		if (net_ratelimit())
			netif_warn(axdev, rx_err, netdev,
				   "maybe reset is needed?\n");
		break;
	default:
		if (net_ratelimit())
			netif_warn(axdev, rx_err, netdev,
				   "Rx status %d\n", status);
		break;
	}

	ax_submit_rx(axdev, desc, GFP_ATOMIC);
}

void ax_write_bulk_callback(struct urb *urb)
{
	struct net_device_stats *stats;
	struct net_device *netdev;
	struct tx_desc *desc;
	struct ax_device *axdev;
	int status = urb->status;

	desc = urb->context;
	if (!desc)
		return;

	axdev = desc->context;
	if (!axdev)
		return;

	netdev = axdev->netdev;
	stats = ax_get_stats(netdev);
	if (status) {
		if (net_ratelimit())
			netif_warn(axdev, tx_err, netdev,
				   "Tx status %d\n", status);
		stats->tx_errors += desc->skb_num;
	} else {
		stats->tx_packets += desc->skb_num;
		stats->tx_bytes += desc->skb_len;
	}

	spin_lock(&axdev->tx_lock);
	list_add_tail(&desc->list, &axdev->tx_free);
	spin_unlock(&axdev->tx_lock);

	usb_autopm_put_interface_async(axdev->intf);

	if (!netif_carrier_ok(netdev))
		return;
	if (!test_bit(AX_ENABLE, &axdev->flags))
		return;
	if (test_bit(AX_UNPLUG, &axdev->flags))
		return;

	if (ax_check_tx_queue_not_empty(axdev) >= 0) 
		napi_schedule(&axdev->napi);
}

static void ax_intr_callback(struct urb *urb)
{
	struct ax_device *axdev;
	struct ax_device_int_data *event = NULL;
	int status = urb->status;
	int res;

	axdev = urb->context;
	if (!axdev)
		return;

	if (!test_bit(AX_ENABLE, &axdev->flags))
		return;
	if (test_bit(AX_UNPLUG, &axdev->flags))
		return;

	switch (status) {
	case 0:
		break;
	case -ECONNRESET:
	case -ESHUTDOWN:
		netif_device_detach(axdev->netdev);
		netif_info(axdev, intr, axdev->netdev,
			   "Stop submitting intr, status %d\n", status);
		return;
	case -ENOENT:
		netif_info(axdev, intr, axdev->netdev,
			   "Stop submitting intr, status %d\n", status);
		return;
	case -EPROTO:
		netif_info(axdev, intr, axdev->netdev,
			   "Stop submitting intr, status %d\n", status);
		return;
	case -EOVERFLOW:
		netif_info(axdev, intr, axdev->netdev,
			   "intr status -EOVERFLOW\n");
		goto resubmit;
	default:
		netif_info(axdev, intr, axdev->netdev,
			   "intr status %d\n", status);
		goto resubmit;
	}
	
	event = urb->transfer_buffer;
	axdev->link = event->link & AX_INT_PPLS_LINK;

	if (axdev->link) {
		if (!netif_carrier_ok(axdev->netdev)) {
			axdev->int_link_info = event->link_info;
			set_bit(AX_LINK_CHG, &axdev->flags);
			schedule_delayed_work(&axdev->schedule, 0);
		}
	} else {
		if (netif_carrier_ok(axdev->netdev)) {
			axdev->int_link_info = 0;
			netif_stop_queue(axdev->netdev);
			set_bit(AX_LINK_CHG, &axdev->flags);
			schedule_delayed_work(&axdev->schedule, 0);
		}
	}

resubmit:
	res = usb_submit_urb(urb, GFP_ATOMIC);
	if (res == -ENODEV) {
		ax_set_unplug(axdev);
		netif_device_detach(axdev->netdev);
	} else if (res) {
		netif_err(axdev, intr, axdev->netdev,
			  "can't resubmit intr, status %d\n", res);
	}
}

inline void *__rx_buf_align(void *data)
{
	return (void *)ALIGN((uintptr_t)data, RX_ALIGN);
}

inline void *__tx_buf_align(void *data, u8 tx_align_len)
{
	return (void *)ALIGN((uintptr_t)data, tx_align_len);
}

static void ax_free_buffer(struct ax_device *axdev)
{
	int i;

	for (i = 0; i < AX88179_MAX_RX; i++) {
		usb_free_urb(axdev->rx_list[i].urb);
		axdev->rx_list[i].urb = NULL;

		kfree(axdev->rx_list[i].buffer);
		axdev->rx_list[i].buffer = NULL;
		axdev->rx_list[i].head = NULL;
	}

	for (i = 0; i < AX88179_MAX_TX; i++) {
		usb_free_urb(axdev->tx_list[i].urb);
		axdev->tx_list[i].urb = NULL;

		kfree(axdev->tx_list[i].buffer);
		axdev->tx_list[i].buffer = NULL;
		axdev->tx_list[i].head = NULL;
	}

	usb_free_urb(axdev->intr_urb);
	axdev->intr_urb = NULL;

	kfree(axdev->intr_buff);
	axdev->intr_buff = NULL;
}

static int ax_alloc_buffer(struct ax_device *axdev)
{
	struct net_device *netdev = axdev->netdev;
	struct usb_interface *intf = axdev->intf;
	struct usb_host_interface *alt = intf->cur_altsetting;
	struct usb_host_endpoint *ep_intr = alt->endpoint;
	struct urb *urb;
	int node, i;
	u8 *buf;

	node = netdev->dev.parent ? dev_to_node(netdev->dev.parent) : -1;

	spin_lock_init(&axdev->rx_lock);
	spin_lock_init(&axdev->tx_lock);
	INIT_LIST_HEAD(&axdev->tx_free);
	INIT_LIST_HEAD(&axdev->rx_done);
	skb_queue_head_init(&axdev->tx_queue);
	skb_queue_head_init(&axdev->rx_queue);

	for (i = 0; i < AX88179_MAX_RX; i++) {
		buf = kmalloc_node(axdev->driver_info->buf_rx_size,
				   GFP_KERNEL, node);
		if (!buf)
			goto err1;

		if (buf != __rx_buf_align(buf)) {
			kfree(buf);
			buf = kmalloc_node(
				axdev->driver_info->buf_rx_size + RX_ALIGN,
				GFP_KERNEL,
				node);
			if (!buf)
				goto err1;
		}

		urb = usb_alloc_urb(0, GFP_KERNEL);
		if (!urb) {
			kfree(buf);
			goto err1;
		}

		INIT_LIST_HEAD(&axdev->rx_list[i].list);
		axdev->rx_list[i].context = axdev;
		axdev->rx_list[i].urb = urb;
		axdev->rx_list[i].buffer = buf;
		axdev->rx_list[i].head = __rx_buf_align(buf);
	}

	for (i = 0; i < AX88179_MAX_TX; i++) {
		buf = kmalloc_node(AX88179_BUF_TX_SIZE, GFP_KERNEL, node);
		if (!buf)
			goto err1;

		if (buf != __tx_buf_align(buf, axdev->tx_align_len)) {
			kfree(buf);
			buf = kmalloc_node(
				      AX88179_BUF_TX_SIZE + axdev->tx_align_len,
				      GFP_KERNEL, node);
			if (!buf)
				goto err1;
		}

		urb = usb_alloc_urb(0, GFP_KERNEL);
		if (!urb) {
			kfree(buf);
			goto err1;
		}

		INIT_LIST_HEAD(&axdev->tx_list[i].list);
		axdev->tx_list[i].context = axdev;
		axdev->tx_list[i].urb = urb;
		axdev->tx_list[i].buffer = buf;
		axdev->tx_list[i].head = __tx_buf_align(buf,
							axdev->tx_align_len);

		list_add_tail(&axdev->tx_list[i].list, &axdev->tx_free);
	}

	axdev->intr_urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!axdev->intr_urb)
		goto err1;

	axdev->intr_buff = kmalloc(INTBUFSIZE, GFP_KERNEL);
	if (!axdev->intr_buff)
		goto err1;

	axdev->intr_interval = (int)ep_intr->desc.bInterval;
	usb_fill_int_urb(axdev->intr_urb, axdev->udev,
			 usb_rcvintpipe(axdev->udev, 1), axdev->intr_buff,
			 INTBUFSIZE, ax_intr_callback, axdev,
			 axdev->intr_interval);

	return 0;
err1:
	ax_free_buffer(axdev);
	return -ENOMEM;
}

static struct tx_desc *ax_get_tx_desc(struct ax_device *dev)
{
	struct tx_desc *desc = NULL;
	unsigned long flags;

	if (list_empty(&dev->tx_free))
		return NULL;

	spin_lock_irqsave(&dev->tx_lock, flags);
	if (!list_empty(&dev->tx_free)) {
		struct list_head *cursor;

		cursor = dev->tx_free.next;
		list_del_init(cursor);
		desc = list_entry(cursor, struct tx_desc, list);
	}
	spin_unlock_irqrestore(&dev->tx_lock, flags);

	return desc;
}

static void ax_tx_bottom(struct ax_device *axdev)
{
	const struct driver_info *info = axdev->driver_info;
	int ret;

	do {
		struct tx_desc *desc;

		if (ax_check_tx_queue_not_empty(axdev) < 0)
			break;

		desc = ax_get_tx_desc(axdev);
		if (!desc)
			break;

		ret = info->tx_fixup(axdev, desc);
		if (ret) {
			struct net_device *netdev = axdev->netdev;

			if (ret == -ENODEV) {
				ax_set_unplug(axdev);
				netif_device_detach(netdev);
			} else {
				struct net_device_stats *stats;
				unsigned long flags;

				stats = ax_get_stats(netdev);
				stats->tx_dropped += desc->skb_num;

				spin_lock_irqsave(&axdev->tx_lock, flags);
				list_add_tail(&desc->list, &axdev->tx_free);
				spin_unlock_irqrestore(&axdev->tx_lock, flags);
			}
		}
	} while (ret == 0);
}

static void ax_bottom_half(struct ax_device *axdev)
{
	if (test_bit(AX_UNPLUG, &axdev->flags) ||
	    !test_bit(AX_ENABLE, &axdev->flags) ||
	    !netif_carrier_ok(axdev->netdev))
		return;

	clear_bit(AX_SCHEDULE_NAPI, &axdev->flags);

	ax_tx_bottom(axdev);
}

static int ax_rx_bottom(struct ax_device *axdev, int budget)
{
	unsigned long flags;
	struct list_head *cursor, *next, rx_queue;
	int ret = 0, work_done = 0;
	struct napi_struct *napi = &axdev->napi;
	struct net_device *netdev = axdev->netdev;
	struct net_device_stats *stats = ax_get_stats(netdev);

	if (!skb_queue_empty(&axdev->rx_queue)) {
		while (work_done < budget) {
			struct sk_buff *skb = __skb_dequeue(&axdev->rx_queue);
			unsigned int pkt_len;
			if (!skb)
				break;

			pkt_len = skb->len;
			napi_gro_receive(napi, skb);
			work_done++;
			stats->rx_packets++;
			stats->rx_bytes += pkt_len;
		}
	}

	if (list_empty(&axdev->rx_done))
		return work_done;

	INIT_LIST_HEAD(&rx_queue);
	spin_lock_irqsave(&axdev->rx_lock, flags);
	list_splice_init(&axdev->rx_done, &rx_queue);
	spin_unlock_irqrestore(&axdev->rx_lock, flags);

	list_for_each_safe(cursor, next, &rx_queue) {
		struct rx_desc *desc;

		if (unlikely(skb_queue_len(&axdev->rx_queue) >= 1000))
			break;

		list_del_init(cursor);

		desc = list_entry(cursor, struct rx_desc, list);
		work_done += axdev->driver_info->rx_fixup(axdev,
							  desc,
							  work_done,
							  budget);

		ret = ax_submit_rx(axdev, desc, GFP_ATOMIC);
		if (ret) list_add_tail(&desc->list, next);
	}

	if (!list_empty(&rx_queue)) {
		spin_lock_irqsave(&axdev->rx_lock, flags);
		list_splice_tail(&rx_queue, &axdev->rx_done);
		spin_unlock_irqrestore(&axdev->rx_lock, flags);
	}

	return work_done;
}

static
int ax_submit_rx(struct ax_device *dev, struct rx_desc *desc, gfp_t mem_flags)
{
	int ret;

	if (test_bit(AX_UNPLUG, &dev->flags) ||
	    !test_bit(AX_ENABLE, &dev->flags) ||
	    !netif_carrier_ok(dev->netdev))
		return 0;

	usb_fill_bulk_urb(desc->urb, dev->udev, usb_rcvbulkpipe(dev->udev, 2),
			  desc->head, dev->driver_info->buf_rx_size,
			  (usb_complete_t)ax_read_bulk_callback, desc);

	ret = usb_submit_urb(desc->urb, mem_flags);
	if (ret == -ENODEV) {
		ax_set_unplug(dev);
		netif_device_detach(dev->netdev);
	} else if (ret) {
		struct urb *urb = desc->urb;
		unsigned long flags;

		urb->actual_length = 0;
		spin_lock_irqsave(&dev->rx_lock, flags);
		list_add_tail(&desc->list, &dev->rx_done);
		spin_unlock_irqrestore(&dev->rx_lock, flags);

		netif_err(dev, rx_err, dev->netdev,
			  "Couldn't submit rx[%p], ret = %d\n", desc, ret);

		napi_schedule(&dev->napi);
	}

	return ret;
}

static inline int __ax_poll(struct ax_device *axdev, int budget)
{
	struct napi_struct *napi = &axdev->napi;
	int work_done;

	work_done = ax_rx_bottom(axdev, budget);
	ax_bottom_half(axdev);

	if (work_done < budget) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,10,0)
		napi_complete_done(napi, work_done);
#else
		if (!napi_complete_done(napi, work_done))
			goto out;
#endif
		if (!list_empty(&axdev->rx_done))
			napi_schedule(napi);
		else if (ax_check_tx_queue_not_empty(axdev) >= 0 &&
			 !list_empty(&axdev->tx_free)) 
			napi_schedule(napi);
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,10,0)
out:
#endif
	return work_done;
}

static int ax_poll(struct napi_struct *napi, int budget)
{
	struct ax_device *axdev = container_of(napi, struct ax_device, napi);

	return __ax_poll(axdev, budget);
}

static void ax_drop_queued_tx(struct ax_device *axdev)
{
	struct net_device_stats *stats = ax_get_stats(axdev->netdev);
	struct sk_buff_head skb_head, *tx_queue = &axdev->tx_queue;
	struct sk_buff *skb;

	if (skb_queue_empty(tx_queue))
		return;

	__skb_queue_head_init(&skb_head);
	spin_lock_bh(&tx_queue->lock);
	skb_queue_splice_init(tx_queue, &skb_head);
	spin_unlock_bh(&tx_queue->lock);

	while ((skb = __skb_dequeue(&skb_head))) {
		dev_kfree_skb(skb);
		stats->tx_dropped++;
	}

}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,6,0)
static void ax_tx_timeout(struct net_device *netdev, unsigned int txqueue)
#else
static void ax_tx_timeout(struct net_device *netdev)
#endif
{
	struct ax_device *dev = netdev_priv(netdev);

	netif_warn(dev, tx_err, netdev, "Tx timeout\n");

	usb_queue_reset_device(dev->intf);
}

static netdev_tx_t ax_start_xmit(struct sk_buff *skb, struct net_device *netdev)
{
	struct ax_device *axdev = netdev_priv(netdev);

	skb_tx_timestamp(skb);

	skb_queue_tail(&axdev->tx_queue, skb);

	if (!list_empty(&axdev->tx_free)) {
		if (test_bit(AX_SELECTIVE_SUSPEND, &axdev->flags)) {
			set_bit(AX_SCHEDULE_NAPI, &axdev->flags);
			schedule_delayed_work(&axdev->schedule, 0);
		} else {
			usb_mark_last_busy(axdev->udev);
			napi_schedule(&axdev->napi);
		}
	} else if (ax_check_tx_queue_len(axdev)) {
		netif_stop_queue(netdev);
	}

	return NETDEV_TX_OK;
}

void ax_set_tx_qlen(struct ax_device *dev)
{
	struct net_device *netdev = dev->netdev;

	dev->tx_qlen = AX88179_BUF_TX_SIZE / (netdev->mtu + ETH_FCS_LEN + 8);
}

static int ax_start_rx(struct ax_device *axdev)
{
	int i, ret = 0;

	INIT_LIST_HEAD(&axdev->rx_done);
	for (i = 0; i < AX88179_MAX_RX; i++) {
		INIT_LIST_HEAD(&axdev->rx_list[i].list);
		ret = ax_submit_rx(axdev, &axdev->rx_list[i], GFP_KERNEL);
		if (ret)
			break;
	}

	if (ret && ++i < AX88179_MAX_RX) {
		struct list_head rx_queue;
		unsigned long flags;

		INIT_LIST_HEAD(&rx_queue);

		do {
			struct rx_desc *desc = &axdev->rx_list[i++];
			struct urb *urb = desc->urb;

			urb->actual_length = 0;
			list_add_tail(&desc->list, &rx_queue);
		} while (i < AX88179_MAX_RX);

		spin_lock_irqsave(&axdev->rx_lock, flags);
		list_splice_tail(&rx_queue, &axdev->rx_done);
		spin_unlock_irqrestore(&axdev->rx_lock, flags);
	}

	return ret;
}

static int ax_stop_rx(struct ax_device *axdev)
{
	int i;

	for (i = 0; i < AX88179_MAX_RX; i++)
		usb_kill_urb(axdev->rx_list[i].urb);

	while (!skb_queue_empty(&axdev->rx_queue))
		dev_kfree_skb(__skb_dequeue(&axdev->rx_queue));

	return 0;
}

static void ax_disable(struct ax_device *axdev)
{
	int i;

	ax_drop_queued_tx(axdev);

	if (test_bit(AX_UNPLUG, &axdev->flags))
		return;	

	for (i = 0; i < AX88179_MAX_TX; i++)
		usb_kill_urb(axdev->tx_list[i].urb);

	ax_stop_rx(axdev);
}


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 39)
static int
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 3, 0)
ax88179_set_features(struct net_device *net, netdev_features_t features)
#else
ax88179_set_features(struct net_device *net, u32 features)
#endif

{
	u8 reg8;
	struct ax_device *dev = netdev_priv(net);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 3, 0)
	netdev_features_t changed = net->features ^ features;
#else
	u32 changed = net->features ^ features;
#endif

	if (changed & NETIF_F_IP_CSUM) {
		ax_read_cmd(dev, AX_ACCESS_MAC, AX_TXCOE_CTL, 1, 1, &reg8, 0);
		reg8 ^= AX_TXCOE_TCP | AX_TXCOE_UDP;
		ax_write_cmd(dev, AX_ACCESS_MAC, AX_TXCOE_CTL, 1, 1, &reg8);
	}

	if (changed & NETIF_F_IPV6_CSUM) {
		ax_read_cmd(dev, AX_ACCESS_MAC, AX_TXCOE_CTL, 1, 1, &reg8, 0);
		reg8 ^= AX_TXCOE_TCPV6 | AX_TXCOE_UDPV6;
		ax_write_cmd(dev, AX_ACCESS_MAC, AX_TXCOE_CTL, 1, 1, &reg8);
	}

	if (changed & NETIF_F_RXCSUM) {
		ax_read_cmd(dev, AX_ACCESS_MAC, AX_RXCOE_CTL, 1, 1, &reg8, 0);
		reg8 ^= AX_RXCOE_IP | AX_RXCOE_TCP | AX_RXCOE_UDP |
			AX_RXCOE_TCPV6 | AX_RXCOE_UDPV6;
		ax_write_cmd(dev, AX_ACCESS_MAC, AX_RXCOE_CTL, 1, 1, &reg8);
	}

	return 0;
}
#endif

static void ax_set_carrier(struct ax_device *axdev)
{
	struct net_device *netdev = axdev->netdev;
	struct napi_struct *napi = &axdev->napi;

	if (axdev->link) {
		if (!netif_carrier_ok(netdev)) {
				netif_stop_queue(netdev);
				napi_disable(napi);
				axdev->driver_info->link_reset(axdev);
				netif_carrier_on(netdev);
				ax_start_rx(axdev);
				napi_enable(napi);
				netif_wake_queue(netdev);
		} else if (netif_queue_stopped(netdev) && 
			   ax_check_tx_queue_len(axdev)) {
				netif_wake_queue(netdev);
		}
	} else {
		if (netif_carrier_ok(netdev)) {
				netif_carrier_off(netdev);
				napi_disable(napi);
				ax_disable(axdev);
				napi_enable(napi);
				netif_info(axdev, link, netdev, "link down\n");
		}
	}
}

static inline void __ax_work_func(struct ax_device *axdev)
{
	if (test_bit(AX_UNPLUG, &axdev->flags) || !netif_running(axdev->netdev))
		return;

	if (usb_autopm_get_interface(axdev->intf) < 0)
		return;

	if (!test_bit(AX_ENABLE, &axdev->flags))
		goto out1;

	if (!mutex_trylock(&axdev->control)) {
		schedule_delayed_work(&axdev->schedule, 0);
		goto out1;
	}

	if (test_and_clear_bit(AX_LINK_CHG, &axdev->flags))
		ax_set_carrier(axdev);

	if (test_and_clear_bit(AX_SCHEDULE_NAPI, &axdev->flags) &&
	    netif_carrier_ok(axdev->netdev))
		napi_schedule(&axdev->napi);

	mutex_unlock(&axdev->control);

out1:
	usb_autopm_put_interface(axdev->intf);
}

static void ax_work_func_t(struct work_struct *work)
{
	struct ax_device *axdev = container_of(work,
					       struct ax_device, schedule.work);

	__ax_work_func(axdev);
}

int ax_usb_command(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	struct _ax_usb_command	*usb_cmd = &info->usb_cmd;
	int err = -ENOMEM, timeout;
	u16 size = usb_cmd->size;
	u8 data[16] = {0}, reqtype;
#if 0
	printk("%s - cmd: 0x%x", __func__, usb_cmd->cmd);
	printk("%s - value: 0x%x", __func__, usb_cmd->value);
	printk("%s - index: 0x%x", __func__, usb_cmd->index);
	printk("%s - size: 0x%x", __func__, size);
#endif
	if (usb_cmd->ops == USB_READ_OPS) {
		reqtype = USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE;
		timeout = USB_CTRL_GET_TIMEOUT;
	} else {
		reqtype = USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE;
		err = copy_from_user(data, (void __user *)info->usb_cmd.data,
				     size);
		if (err) {
			printk("%s: copy_from_user failed. (%d)",
				__func__, err);
			return err;
		}
		timeout = USB_CTRL_SET_TIMEOUT;
	}

	err = usb_control_msg(axdev->udev, usb_rcvctrlpipe(axdev->udev, 0),
			      usb_cmd->cmd, reqtype, usb_cmd->value,
			      usb_cmd->index, data, size, timeout);
	if (err > 0 && err <= size) {
		err = copy_to_user((void __user *)info->usb_cmd.data, data,
				   size);
		if (err) {
			printk("%s: copy_to_user failed. (%d)", __func__, err);
			return err;
		}
	}

	return 0;
}

static int ax_open(struct net_device *netdev)
{
	struct ax_device *axdev = netdev_priv(netdev);
	int res = 0;

	res = ax_alloc_buffer(axdev);
	if (res) 
		goto out;

	res = usb_autopm_get_interface(axdev->intf);
	if (res < 0) 
		goto out_free;

	mutex_lock(&axdev->control);

	netif_carrier_off(netdev);

	res = axdev->driver_info->hw_init(axdev);
	if (res < 0)
		goto out_free;

	res = usb_submit_urb(axdev->intr_urb, GFP_KERNEL);
	if (res) {
		if (res == -ENODEV)
			netif_device_detach(netdev);
		netif_warn(axdev, ifup, netdev, "intr_urb submit failed: %d\n",
			   res);
		goto out_unlock;
	}

	smp_mb__before_atomic();
	set_bit(AX_ENABLE, &axdev->flags);
	smp_mb__after_atomic();

	napi_enable(&axdev->napi);

	netif_start_queue(netdev);

	mutex_unlock(&axdev->control);

	usb_autopm_put_interface(axdev->intf);

	return 0;

out_unlock:
	mutex_unlock(&axdev->control);
	usb_autopm_put_interface(axdev->intf);
out_free:
	ax_free_buffer(axdev);
out:
	return res;
}

static int ax_close(struct net_device *netdev)
{
	struct ax_device *axdev = netdev_priv(netdev);
	int ret = 0;

	netif_carrier_off(netdev);

	if (axdev->driver_info->stop)
		axdev->driver_info->stop(axdev);

	napi_disable(&axdev->napi);
	smp_mb__before_atomic();
	clear_bit(AX_ENABLE, &axdev->flags);
	smp_mb__after_atomic();
	usb_kill_urb(axdev->intr_urb);
	cancel_delayed_work_sync(&axdev->schedule);
	netif_stop_queue(axdev->netdev);

	ret = usb_autopm_get_interface(axdev->intf);
	if (ret < 0 || test_bit(AX_UNPLUG, &axdev->flags)) {
		ax_drop_queued_tx(axdev);
		ax_stop_rx(axdev);
	} 

	ax_disable(axdev);

	ax_free_buffer(axdev);

	return ret;
}

static int ax88179_change_mtu(struct net_device *net, int new_mtu)
{
	struct ax_device *axdev = netdev_priv(net);
	u16 reg16;

	if (new_mtu <= 0 || new_mtu > 4088)
		return -EINVAL;

	net->mtu = new_mtu;

	if (net->mtu > 1500) {
		ax_read_cmd(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
				 2, 2, &reg16, 1); /* ??? Should read 2byes in data stage */
		reg16 |= AX_MEDIUM_JUMBO_EN;
		ax_write_cmd(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
				  2, 2, &reg16);
	} else {
		ax_read_cmd(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
				 2, 2, &reg16, 1);
		reg16 &= ~AX_MEDIUM_JUMBO_EN;
		ax_write_cmd(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
				  2, 2, &reg16);
	}

	return 0;
}

int ax_check_ether_addr(struct ax_device *axdev)
{
	unsigned char *tmp = (unsigned char*)axdev->netdev->dev_addr;
	u8 default_mac[6] = {0, 0x0e, 0xc6, 0x81, 0x79, 0x01};
	u8 default_mac_178a[6] = {0, 0x0e, 0xc6, 0x81, 0x78, 0x01};

	if (((*((u8*)tmp) == 0) &&
	    (*((u8*)tmp + 1) == 0) &&
	    (*((u8*)tmp + 2) == 0)) ||
	    !is_valid_ether_addr((u8*)tmp) ||
	    !memcmp(axdev->netdev->dev_addr, default_mac, ETH_ALEN) ||
	    !memcmp(axdev->netdev->dev_addr, default_mac_178a, ETH_ALEN)) {
		eth_random_addr(tmp);
		*tmp = 0;
		*(tmp + 1) = 0x0E;
		*(tmp + 2) = 0xC6;

		return -EADDRNOTAVAIL;
	}

	return 0;
}

static int ax_select_chip_mode(struct ax_device *axdev)
{	
	switch (axdev->chip_version) {
	case AX_VERSION_AX88179:
		axdev->sub_version = 0;
		return 0;
	case AX_VERSION_AX88179A_772D:

		switch (axdev->sub_version) {
		case 2:
			axdev->int_link_speed = true;
			break;
		default:
			axdev->int_link_speed = false;
			break;
		};
		break;
	case AX_VERSION_AX88279:
		axdev->sub_version = 0;
		axdev->int_link_speed = true;
		break;
	default:
		return -ENODEV;
	};

	return 0;
}

static int ax_get_chip_feature(struct ax_device *axdev)
{
	int ret = 0;

	axdev->chip_version = AX_VERSION_INVALID;	

	switch (axdev->driver_info->chip_mode) {
	case AX_CHIP_AX88772D:
	case AX_CHIP_AX88179A:
	case AX_CHIP_AX88279:
		ret = ax_read_cmd(axdev, AX_ACCESS_MAC, AX_CHIP_STATUS,
				   1, 1, &axdev->chip_version, 0);
		if (ret < 0)
			return ret;

		axdev->chip_version =
			(axdev->chip_version & AX_CHIP_CODE_MASK) >> 4;

		ret = ax_read_cmd(axdev, AX88179A_ACCESS_BL,
				  AX88179A_HW_EC_VERSION, 1, 1,
				  &axdev->sub_version, 0);
		if (ret < 0) {
			axdev->sub_version = ~0;
			ret = 0;
		}

		ret = ax_read_cmd(axdev, AX88179A_ACCESS_BL,
				  AX88179A_SW_REVERSION, 1, 1,
				  &axdev->fr_rversion, 0);
		if (ret < 0) {
			axdev->fr_rversion = ~0;
			ret = 0;
		} else {
			axdev->fr_rversion &= 0xF;
		}

		ret = ax_select_chip_mode(axdev);
		break;
	case AX_CHIP_AX88179:
		axdev->chip_version = AX_VERSION_AX88179;
		break;
	default:
		return -ENODEV;
	};

	if (axdev->chip_version < AX_VERSION_AX88179)
		return -ENODEV;

	return ret;
}

static int ax_probe(struct usb_interface *intf, const struct usb_device_id *id)
{
	struct usb_device *udev = interface_to_usbdev(intf);
	//struct usb_driver *driver = to_usb_driver(intf->dev.driver);
	const struct driver_info *info;
	struct net_device *netdev;
	struct ax_device *axdev;
	int ret;

	if (udev->actconfig->desc.bConfigurationValue != 1) {
		usb_driver_set_configuration(udev, 1);
		return -ENODEV;
	}

	info = (const struct driver_info *)id->driver_info;
	if(!info || !info->bind || !info->unbind) {
		dev_err(&intf->dev, "Driver method not registered\n");
		return -ENODEV;
	}

	netdev = alloc_etherdev(sizeof(struct ax_device));
	if (!netdev) {
		dev_err(&intf->dev, "Out of memory\n");
		return -ENOMEM;
	}
	
	axdev = netdev_priv(netdev);
	axdev->driver_info = info;

	netdev->watchdog_timeo = AX_TX_TIMEOUT;

	axdev->udev = udev;
	axdev->netdev = netdev;
	axdev->intf = intf;
	intf->needs_remote_wakeup = true;

	axdev->bEEE = bEEE;

	mutex_init(&axdev->control);
	INIT_DELAYED_WORK(&axdev->schedule, ax_work_func_t);

	ret = ax_get_chip_feature(axdev);
	if (ret) {
		dev_err(&intf->dev, "Failed to get Device feature\n");
		goto out;
	}
	ret = info->bind(axdev);
	if (ret) {
		dev_err(&intf->dev, "Device initialization failed\n");
		goto out;
	}

	usb_set_intfdata(intf, axdev);
	netif_napi_add(netdev, &axdev->napi, ax_poll, info->napi_weight);

	SET_NETDEV_DEV(netdev, &intf->dev);
	ret = register_netdev(netdev);
	if (ret != 0) {
		netif_err(axdev, probe, netdev,
			  "couldn't register the device\n");
		goto out1;
	}

	/* usb_enable_autosuspend(udev); */

	return 0;
out1:
	netif_napi_del(&axdev->napi);
	usb_set_intfdata(intf, NULL);
out:
	free_netdev(netdev);
	return ret;
}

static void ax_disconnect(struct usb_interface *intf)
{
	struct ax_device *axdev = usb_get_intfdata(intf);

	usb_set_intfdata(intf, NULL);
	if (axdev) {
		axdev->driver_info->unbind(axdev);
		ax_set_unplug(axdev);
		netif_napi_del(&axdev->napi);
		unregister_netdev(axdev->netdev);
		free_netdev(axdev->netdev);
	}
}

static int ax_pre_reset(struct usb_interface *intf)
{
	struct ax_device *axdev = usb_get_intfdata(intf);
	struct net_device *netdev;

	if (!axdev)
		return 0;

	netdev = axdev->netdev;
	if (!netif_running(netdev))
		return 0;

	netif_stop_queue(netdev);
	napi_disable(&axdev->napi);
	smp_mb__before_atomic();
	clear_bit(AX_ENABLE, &axdev->flags);
	smp_mb__after_atomic();
	usb_kill_urb(axdev->intr_urb);
	cancel_delayed_work_sync(&axdev->schedule);

	return 0;
}

static int ax_post_reset(struct usb_interface *intf)
{
	struct ax_device *axdev = usb_get_intfdata(intf);
	struct net_device *netdev;

	if (!axdev)
		return 0;

	netdev = axdev->netdev;
	if (!netif_running(netdev))
		return 0;

	smp_mb__before_atomic();
	set_bit(AX_ENABLE, &axdev->flags);
	smp_mb__after_atomic();
	if (netif_carrier_ok(netdev)) {
		mutex_lock(&axdev->control);
		ax_start_rx(axdev);
		mutex_unlock(&axdev->control);
	}

	napi_enable(&axdev->napi);
	netif_wake_queue(netdev);
	usb_submit_urb(axdev->intr_urb, GFP_KERNEL);

	if (!list_empty(&axdev->rx_done))
		napi_schedule(&axdev->napi);

	return 0;
}

static int ax_system_resume(struct ax_device *axdev)
{
	struct net_device *netdev = axdev->netdev;

	netif_device_attach(netdev);

	if (netif_running(netdev) && (netdev->flags & IFF_UP)) {
		netif_carrier_off(netdev);

		axdev->driver_info->system_resume(axdev);
		
		smp_mb__before_atomic();
		set_bit(AX_ENABLE, &axdev->flags);
		smp_mb__after_atomic();

		usb_submit_urb(axdev->intr_urb, GFP_NOIO);
	}

	return 0;
}

static int ax_runtime_resume(struct ax_device *axdev)
{
	struct net_device *netdev = axdev->netdev;

	if (netif_running(netdev) && (netdev->flags & IFF_UP)) {
		struct napi_struct *napi = &axdev->napi;

		napi_disable(napi);
		smp_mb__before_atomic();
		set_bit(AX_ENABLE, &axdev->flags);
		smp_mb__after_atomic();

		if (netif_carrier_ok(netdev)) {
			if (axdev->link) {
				axdev->driver_info->link_reset(axdev);
				ax_start_rx(axdev);
			} else {
				netif_carrier_off(netdev);
				ax_disable(axdev);
			}
		}
		
		napi_enable(napi);
		clear_bit(AX_SELECTIVE_SUSPEND, &axdev->flags);
		smp_mb__after_atomic();
		if (!list_empty(&axdev->rx_done)) {
			local_bh_disable();
			napi_schedule(&axdev->napi);
			local_bh_enable();
		}
		ax_write_cmd_nopm(axdev, AX_PHY_POLLING, 1, 0, 0, NULL);
		usb_submit_urb(axdev->intr_urb, GFP_NOIO);
	} else {
		clear_bit(AX_SELECTIVE_SUSPEND, &axdev->flags);
		smp_mb__after_atomic();
	}

	return 0;
}

static int ax_system_suspend(struct ax_device *axdev)
{
	struct net_device *netdev = axdev->netdev;
	int ret = 0;

	netif_device_detach(netdev);

	if (netif_running(netdev) && test_bit(AX_ENABLE, &axdev->flags)) {
		struct napi_struct *napi = &axdev->napi;

		smp_mb__before_atomic();
		clear_bit(AX_ENABLE, &axdev->flags);
		smp_mb__after_atomic();
		usb_kill_urb(axdev->intr_urb);
		ax_disable(axdev);

		axdev->driver_info->system_suspend(axdev);

		napi_disable(napi);
		cancel_delayed_work_sync(&axdev->schedule);
		napi_enable(napi);
	}

	return ret;
}

static int ax_runtime_suspend(struct ax_device *axdev)
{
	struct net_device *netdev = axdev->netdev;
	int ret = 0;

	set_bit(AX_SELECTIVE_SUSPEND, &axdev->flags);
	smp_mb__after_atomic();

	if (netif_running(netdev) && test_bit(AX_ENABLE, &axdev->flags)) {
		u16 reg16;

		if (netif_carrier_ok(netdev)) {
			ax_read_cmd_nopm(axdev, AX_ACCESS_MAC,
					      AX_RX_FREE_BUF_LOW, 
					      2, 2, &reg16, 1);
			if (reg16 != 0x067F) {
				ret = -EBUSY;
				goto out1;
			}
		}

		smp_mb__before_atomic();
		clear_bit(AX_ENABLE, &axdev->flags);
		smp_mb__after_atomic();
		usb_kill_urb(axdev->intr_urb);

		if (netif_carrier_ok(netdev)) {
			struct napi_struct *napi = &axdev->napi;

			napi_disable(napi);
			ax_stop_rx(axdev);
			napi_enable(napi);
		}

		/* Disable RX path */
		ax_read_cmd_nopm(axdev, AX_ACCESS_MAC,
				      AX_MEDIUM_STATUS_MODE, 2, 2, &reg16, 1);
		reg16 &= ~AX_MEDIUM_RECEIVE_EN;
		ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
				       AX_MEDIUM_STATUS_MODE, 2, 2, &reg16);

		/* Configure RX control register => stop operation */
		reg16 = AX_RX_CTL_STOP;
		ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_RX_CTL,
				       2, 2, &reg16);
	}

out1:
	return ret;
}

static int ax_suspend(struct usb_interface *intf, pm_message_t message)
{
	struct ax_device *axdev = usb_get_intfdata(intf);
	int ret;
	
	mutex_lock(&axdev->control);

	if (PMSG_IS_AUTO(message))
		ret = ax_runtime_suspend(axdev);
	else
		ret = ax_system_suspend(axdev);

	mutex_unlock(&axdev->control);

	return ret;
}

static int ax_resume(struct usb_interface *intf)
{
	struct ax_device *axdev = usb_get_intfdata(intf);
	int ret;

	mutex_lock(&axdev->control);

	if (test_bit(AX_SELECTIVE_SUSPEND, &axdev->flags))
		ret = ax_runtime_resume(axdev);
	else
		ret = ax_system_resume(axdev);

	mutex_unlock(&axdev->control);
	return ret;
}

static int ax_reset_resume(struct usb_interface *intf)
{
	struct ax_device *axdev = usb_get_intfdata(intf);

	clear_bit(AX_SELECTIVE_SUSPEND, &axdev->flags);
	mutex_lock(&axdev->control);
	axdev->driver_info->hw_init(axdev);
	mutex_unlock(&axdev->control);

	return 0;
}

const struct net_device_ops ax88179_netdev_ops = {
	.ndo_open		= ax_open,
	.ndo_stop		= ax_close,
	.ndo_do_ioctl		= ax88179_ioctl,
	.ndo_start_xmit		= ax_start_xmit,
	.ndo_tx_timeout		= ax_tx_timeout,
	.ndo_set_features	= ax88179_set_features,
	.ndo_set_rx_mode	= ax88179_set_multicast,
	.ndo_set_mac_address	= ax88179_set_mac_addr,
	.ndo_change_mtu		= ax88179_change_mtu,
	.ndo_validate_addr	= eth_validate_addr,
};

const struct net_device_ops ax88179a_netdev_ops = {
	.ndo_open		= ax_open,
	.ndo_stop		= ax_close,
	.ndo_do_ioctl		= ax88179a_ioctl,
	.ndo_start_xmit		= ax_start_xmit,
	.ndo_tx_timeout		= ax_tx_timeout,
	.ndo_set_features	= ax88179_set_features,
	.ndo_set_rx_mode	= ax88179a_set_multicast,
	.ndo_set_mac_address	= ax88179_set_mac_addr,
	.ndo_change_mtu		= ax88179_change_mtu,
	.ndo_validate_addr	= eth_validate_addr,
};

#define ASIX_USB_DEVICE(vend, prod, lo, hi, info)	\
	.match_flags = USB_DEVICE_ID_MATCH_DEVICE_AND_VERSION | \
		       USB_DEVICE_ID_MATCH_INT_CLASS, \
	.idVendor = (vend), \
	.idProduct = (prod), \
	.bcdDevice_lo = (lo), \
	.bcdDevice_hi = (hi), \
	.bInterfaceClass = USB_CLASS_VENDOR_SPEC, \
	.driver_info = (unsigned long)&info \
}, \
{ \
	.match_flags = USB_DEVICE_ID_MATCH_INT_INFO | \
		       USB_DEVICE_ID_MATCH_DEVICE_AND_VERSION, \
	.idVendor = (vend), \
	.idProduct = (prod), \
	.bcdDevice_lo = (lo), \
	.bcdDevice_hi = (hi), \
	.bInterfaceClass = USB_CLASS_COMM, \
	.bInterfaceSubClass = USB_CDC_SUBCLASS_ETHERNET, \
	.bInterfaceProtocol = USB_CDC_PROTO_NONE, \
	.driver_info = (unsigned long)&info \
}, \
{ \
	.match_flags = USB_DEVICE_ID_MATCH_INT_INFO | \
		       USB_DEVICE_ID_MATCH_DEVICE_AND_VERSION, \
	.idVendor = (vend), \
	.idProduct = (prod), \
	.bcdDevice_lo = (lo), \
	.bcdDevice_hi = (hi), \
	.bInterfaceClass = USB_CLASS_COMM, \
	.bInterfaceSubClass = USB_CDC_SUBCLASS_NCM, \
	.bInterfaceProtocol = USB_CDC_PROTO_NONE, \
	.driver_info = (unsigned long)&info

static const struct usb_device_id ax_usb_table[] = {
	{
		USB_DEVICE_VER(USB_VENDOR_ID_ASIX, AX_DEVICE_ID_179X,
			       0x0000, AX_BCDDEVICE_ID_179),
		.driver_info = (unsigned long)&ax88179_info,
	},
	{
		USB_DEVICE_VER(USB_VENDOR_ID_ASIX, AX_DEVICE_ID_178A,
			       0x0000, AX_BCDDEVICE_ID_178A),
		.driver_info = (unsigned long)&ax88179_info,
	},
	{
		USB_DEVICE_VER(0x0df6, 0x0072,
			       0x0000, AX_BCDDEVICE_ID_179),
		.driver_info = (unsigned long)&ax88179_info,
	},
	{
		USB_DEVICE_VER(0x17ef, 0x304b,
			       0x0000, AX_BCDDEVICE_ID_179),
		.driver_info = (unsigned long)&ax88179_info,
	},
	{
		USB_DEVICE_VER(0x0930, 0x0a13,
			       0x0000, AX_BCDDEVICE_ID_179),
		.driver_info = (unsigned long)&ax88179_info,
	},
	{
		USB_DEVICE_VER(0x04e8, 0xa100,
			       0x0000, AX_BCDDEVICE_ID_179),
		.driver_info = (unsigned long)&ax88179_info,
	},
	{
		USB_DEVICE_VER(0x2001, 0x4a00,
			       0x0000, AX_BCDDEVICE_ID_179),
		.driver_info = (unsigned long)&ax88179_info,
	},
	{
		USB_DEVICE_VER(0x0711, 0x0179,
			       0x0000, AX_BCDDEVICE_ID_179),
		.driver_info = (unsigned long)&ax88179_info,
	},
	{
		ASIX_USB_DEVICE(USB_VENDOR_ID_ASIX, AX_DEVICE_ID_179X,
				0x0000, AX_BCDDEVICE_ID_772D, ax88179a_info),
	},
	{
		ASIX_USB_DEVICE(USB_VENDOR_ID_ASIX, AX_DEVICE_ID_179X,
				0x0000, AX_BCDDEVICE_ID_179A, ax88179a_info),
	},
	{
		ASIX_USB_DEVICE(USB_VENDOR_ID_ASIX, AX_DEVICE_ID_179X,
				0x0000, AX_BCDDEVICE_ID_279, ax88179a_info),
	},

	{}
};

MODULE_DEVICE_TABLE(usb, ax_usb_table);

static struct usb_driver ax_usb_driver = {
	.name		= MODULENAME,
	.id_table	= ax_usb_table,
	.probe		= ax_probe,
	.disconnect	= ax_disconnect,
	.suspend	= ax_suspend,
	.resume		= ax_resume,
	.reset_resume	= ax_reset_resume,
	.pre_reset	= ax_pre_reset,
	.post_reset 	= ax_post_reset,
	.supports_autosuspend = 1,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,5,0)
	.disable_hub_initiated_lpm = 1,
#endif
};

module_usb_driver(ax_usb_driver);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");
MODULE_VERSION(DRIVER_VERSION);
