#include "ax_main.h"
#include "ax88279_179a_772d.h"

static struct {unsigned char ctrl, timer_l, timer_h, size, ifg; }
AX88179A_BULKIN_SIZE[] =	{
	{5, 0xF0, 0x00,	0x0C, 0x0F},	//1G, SS
	{5, 0xC0, 0x02,	0x06, 0x0F},	//1G, HS
	{7, 0xF0, 0x00,	0x0C, 0x0F},	//100M, Full, SS
	{6, 0x00, 0x00,	0x06, 0x0F},	//100M, Half, SS
	{5, 0xC0, 0x04,	0x06, 0x0F},	//100M, Full, HS
	{7, 0xC0, 0x04,	0x06, 0x0F},	//100M, Half, HS
	{7, 0x00, 0,	0x03, 0x3F},	//FS
};

static struct {unsigned char ctrl, timer_l, timer_h, size, ifg; }
AX88279_BULKIN_SIZE[] =	{
	{5, 0x10, 0x01,	0x11, 0x0F},	//2.5G
	{7, 0xB3, 0x01,	0x11, 0x0F},	//1G, SS
	{7, 0xC0, 0x02,	0x06, 0x0F},	//1G, HS
	{7, 0x80, 0x01,	0x03, 0x0F},	//100M, Full, SS
	{7, 0x80, 0x01,	0x03, 0x0F},	//100M, Half, SS
	{7, 0x80, 0x01,	0x03, 0x0F},	//100M, Full, HS
	{7, 0x80, 0x01,	0x03, 0x0F},	//100M, Half, HS
	{7, 0x00, 0,	0x03, 0x3F},	//FS
};

const struct ethtool_ops ax88179a_ethtool_ops = {
	.get_drvinfo  	= ax_get_drvinfo,
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 10, 0)
	.get_settings 	= ax_get_settings,
	.set_settings 	= ax_set_settings,
#else
	.get_link_ksettings = ax_get_link_ksettings,
	.set_link_ksettings = ax_set_link_ksettings,
#endif
	.get_link     	= ethtool_op_get_link,
	.get_msglevel 	= ax_get_msglevel,
	.set_msglevel 	= ax_set_msglevel,
	.get_wol 	= ax_get_wol,
	.set_wol 	= ax_set_wol,
};

void ax88179a_get_fw_version(struct ax_device *axdev)
{
	int i;

	for (i = 0; i < 3; i++) {
		if (ax_read_cmd(axdev, AX88179A_ACCESS_BL, (0xFD + i),
				1, 1, &axdev->fw_version[i], 1) < 0) {
			axdev->fw_version[i] = ~0;
		}
	}
}

int ax88179a_signature(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	strncpy(info->sig, AX88179A_SIGNATURE, strlen(AX88179A_SIGNATURE));
	return 0;
}

int ax88179a_read_version(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	unsigned char temp[16] = {0};
	u8 *version = axdev->fw_version;

	sprintf(temp, "v%d.%d.%d", version[0], version[1], version[2]);

	memcpy(&info->version.version, temp, 16);

	return 0;
}

int ax88179a_write_flash(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	int i, ret;
	u8 *buf = NULL;

	buf = (u8 *)kmalloc(256, GFP_KERNEL);
	if (!buf) {
		netdev_err(axdev->netdev,
			   "Cannot allocate memory for buffer");
		return -ENOMEM;
	}

	ret = ax_write_cmd(axdev, AX88179A_FLASH_WEN, 0, 0, 0, NULL);
	if (ret < 0) {
		printk("USB command flash write enable failed");
		info->flash.status = -ERR_FALSH_WRITE_EN;
		goto out;
	}
	
	for (i = info->flash.offset;
	     i < (info->flash.length + info->flash.offset);
	     i += 256) {
		if (copy_from_user(buf,
				   (void __user *)&info->flash.buf[i], 256)) {
			ret = -EFAULT;
			goto out;
		}

		ret = ax_write_cmd(axdev, AX88179A_FLASH_WRITE,
					(u16)((i >> 16) & 0xFFFF),
					(u16)(i & 0xFFFF), 256, buf);
		if (ret < 0) {
			info->flash.status = -ERR_FALSH_WRITE;
			goto out;
		}
	}

	ret = ax_write_cmd(axdev, AX88179A_FLASH_WDIS, 0, 0, 0, NULL);
	if (ret < 0) {
		printk("USB command flash write disable failed");
		info->flash.status = -ERR_FALSH_WRITE_DIS;
		return ret;
	}
out:
	if (buf)
		kfree(buf);
	return ret;
}

int ax88179a_read_flash(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	int i, ret = 0;
	u8 *buf = NULL;

	buf = (u8 *)kmalloc(256, GFP_KERNEL);
	if (!buf) {
		netdev_err(axdev->netdev,
			   "Cannot allocate memory for buffer");
		return -ENOMEM;
	}

	for (i = info->flash.offset;
	     i < (info->flash.length + info->flash.offset);
	     i += 256) {
		ret = ax_read_cmd(axdev, AX88179A_FLASH_READ,
				       (u16)((i >> 16) & 0xFFFF),
				       (u16)(i & 0xFFFF),
				       256, buf, 0);
		if (ret < 0) {
			info->flash.status = -ERR_FALSH_READ;
			break;
		}

		if (copy_to_user((void __user *)&info->flash.buf[i],
				 buf, 256)) {
			ret = -EFAULT;
			break;
		}
	}

	if (buf)
		kfree(buf);
	return ret;
}

void dump_efuse_data(u8 *efuse)
{
#if 0
	int i;
	unsigned char *data = (unsigned char *)efuse;

	for (i = 0; i < (20); i += 4) {
		printk("%02X %02X %02X %02X",
			data[i + 3], data[i + 2], data[i + 1], data[i]);
	}
#endif
}

int ax88179a_program_efuse(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	int ret = 0;
	u16 offset = (u16)(info->flash.offset * 16);
	u8 buf[20] = {0};

	ret = copy_from_user(buf, (void __user *)info->flash.buf, 20);
	if (ret) {
		printk("%s: copy_from_user failed. (%d)", __func__, ret);
		return ret;
	}

	dump_efuse_data(buf);

	ret = ax_write_cmd(axdev, AX_ACCESS_EFUSE, offset, 0, 20, buf);
	if (ret < 0) {
		info->flash.status = -ERR_EFUSE_WRITE;
		return ret;
	}

	return ret;
}

int ax88179a_dump_efuse(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	int ret = 0;
	u16 offset = (u16)(info->flash.offset * 16);
	u8 buf[20] = {0};

	ret = ax_read_cmd(axdev, AX_ACCESS_EFUSE, offset, 0, 20, buf, 0);
	if (ret < 0) {
		info->flash.status = -ERR_EFUSE_READ;
		return ret;
	}

	dump_efuse_data(buf);

	ret = copy_to_user((void __user *)info->flash.buf, buf, 20);
	if (ret)
		printk("%s: copy_to_user failed. (%d)", __func__, ret);

	return ret;
}

int ax88179a_boot_to_rom(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	usb_control_msg(axdev->udev, usb_sndctrlpipe(axdev->udev, 0),
			AX88179A_BOOT_TO_ROM,
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			0x5A5A, 0xA5A5, NULL, 0, 1);

	return 0;
}

int ax88179a_erase_flash(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	int ret = 0;

	ret = ax_write_cmd(axdev, AX88179A_FLASH_WEN, 0, 0, 0, NULL);
	if (ret < 0) {
		printk("USB command flash write enable failed");
		info->flash.status = -ERR_FALSH_WRITE_EN;
		return ret;
	}

	ret = usb_control_msg(axdev->udev, usb_sndctrlpipe(axdev->udev, 0),
			      AX88179A_FLASH_EARSE_ALL,
			      USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			      0, 0, NULL, 0, 300000);
	if (ret < 0) {
		printk("USB command flash erase all failed");
		info->flash.status = -ERR_FALSH_ERASE_ALL;
		return ret;
	}

	ret = ax_write_cmd(axdev, AX88179A_FLASH_WDIS, 0, 0, 0, NULL);
	if (ret < 0) {
		printk("USB command flash write disable failed");
		info->flash.status = -ERR_FALSH_WRITE_DIS;
		return ret;
	}

	return 0;
}

int ax88179a_sw_reset(struct ax_device *axdev, AX_IOCTL_COMMAND *info)
{
	u32 *buf = NULL;

	buf = (u32 *)kmalloc(4, GFP_KERNEL);
	if (!buf) {
		netdev_err(axdev->netdev,
			   "Cannot allocate memory for buffer");
		return -ENOMEM;
	}
	
	*buf = 1;

	usb_control_msg(axdev->udev, usb_sndctrlpipe(axdev->udev, 0), 0x10,
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			0x18E8, 0x000F, buf, 4, 1);

	kfree(buf);

	return 0;
}

IOCTRL_TABLE ax88179a_tbl[] = {
	ax88179a_signature,
	ax_usb_command,
	ax88179a_read_version,
	ax88179a_write_flash,
	ax88179a_boot_to_rom,
	ax88179a_erase_flash,
	ax88179a_sw_reset,
	ax88179a_read_flash,
	ax88179a_program_efuse,
	ax88179a_dump_efuse,
};

int ax88179a_ioctl(struct net_device *net, struct ifreq *rq, int cmd)
{
	struct ax_device *axdev = netdev_priv(net);
	AX_IOCTL_COMMAND info;
	void __user *uptr = (void __user *) rq->ifr_data;
	int private_cmd, ret;

	switch (cmd) { 
	case AX_PRIVATE:
	{
		if (copy_from_user(&info, uptr, sizeof(AX_IOCTL_COMMAND))) {
			printk(KERN_INFO "copy_from_user, return -EFAULT");
			return -EFAULT;
		}

		private_cmd = info.ioctl_cmd;
		ret = (*ax88179a_tbl[private_cmd])(axdev, &info);
		if (ret < 0) {
			printk(KERN_INFO "ax88179a_tbl, return %d", ret);
			return ret;
		}

		if (copy_to_user(uptr, &info, sizeof(AX_IOCTL_COMMAND))) {
			printk(KERN_INFO "copy_to_user, return -EFAULT");
			return -EFAULT;
		}

		break;
	}
	default : 
		return  generic_mii_ioctl(&axdev->mii, if_mii(rq), cmd, NULL);
	}
	return 0;
}

static int ax88179a_get_mac(struct ax_device *axdev)
{
	int ret;

	ret = ax_read_cmd(axdev, AX_ACCESS_MAC, AX_NODE_ID, ETH_ALEN,
			       ETH_ALEN, axdev->netdev->dev_addr, 0);
	if (ret < 0) {
		netdev_err(axdev->netdev,
			   "Failed to read MAC address: %d", ret);
		return ret;
	}

	if (ax_check_ether_addr(axdev)) {
		netdev_warn(axdev->netdev,
			    "Found invalid MAC address value");
	}

	memcpy(axdev->netdev->perm_addr, axdev->netdev->dev_addr, ETH_ALEN);

	ax_write_cmd(axdev, AX_ACCESS_MAC, AX_NODE_ID, ETH_ALEN,
			  ETH_ALEN, axdev->netdev->dev_addr);
	if (ret < 0) {
		netdev_err(axdev->netdev,
			   "Failed to write MAC address: %d", ret);
		return ret;
	}


	return 0;
}

static bool ax88179a_check_phy_power(struct ax_device *axdev)
{
	u8 reg8 = 0;
	int ret = 0;

	ret = ax_read_cmd_nopm(axdev, AX88179A_PHY_POWER, 0, 0, 1, &reg8, 1);
	if(ret < 0)
		return false;

	return (reg8 & AX_PHY_POWER);
}

static bool ax88179a_set_phy_power(struct ax_device *axdev, bool on)
{
	u8 reg8;
	int ret;

	reg8 = (on)?AX_PHY_POWER:0;
	ret = ax_write_cmd_nopm(axdev, AX88179A_PHY_POWER, 0, 0, 1, &reg8);
	if(ret < 0)
		return ret;
	msleep(250);

	return 0;
}

static void ax88179a_Gether_setting(struct ax_device *axdev)
{
	u16 reg16;

	ax_write_cmd(axdev, AX88179_GPHY_CTRL, 1, 1, 0, NULL);

	reg16 = GMII_SELECTOR_8023 | GMII_ANLPAR_ASYM_PAUSE |
		GMII_ANLPAR_PAUSE | GMII_ANLPAR_10T |
		GMII_ANLPAR_10TFD | GMII_ANLPAR_100TX | GMII_ANLPAR_100TXFD;
	ax_write_cmd(axdev, AX_ACCESS_PHY, AX88179A_PHY_ID, 
		     GMII_PHY_ANAR, 2, &reg16);

	reg16 = GMII_1000_AUX_CTRL_FD_CAPABLE;
	ax_write_cmd(axdev, AX_ACCESS_PHY, AX88179A_PHY_ID, 
		     GMII_PHY_1000BT_CONTROL, 2, &reg16);
}

static int ax88179a_bind(struct ax_device *axdev)
{
	struct net_device *netdev = axdev->netdev;
	int ret;

	ax88179a_get_fw_version(axdev);

	PRINT_VERSION(axdev, AX_DRIVER_STRING_179A_772D);

	ret = ax_write_cmd(axdev, AX_FW_MODE, AX_FW_MODE_179A, 0, 0, NULL);
	if (ret < 0)
		return ret;

	ret = ax_write_cmd(axdev, AX_RELOAD_FLASH_EFUSE, 0, 0, 0, NULL);
	if (ret < 0)
		return ret;

	netdev->features    |= NETIF_F_IP_CSUM | NETIF_F_IPV6_CSUM |
			       NETIF_F_SG | NETIF_F_TSO | NETIF_F_FRAGLIST |
			       NETIF_F_HW_VLAN_CTAG_RX |
			       NETIF_F_HW_VLAN_CTAG_TX;
	netdev->hw_features |= NETIF_F_IP_CSUM | NETIF_F_IPV6_CSUM |
			       NETIF_F_SG | NETIF_F_TSO | NETIF_F_FRAGLIST |
			       NETIF_F_HW_VLAN_CTAG_RX |
			       NETIF_F_HW_VLAN_CTAG_TX;
	netdev->vlan_features = NETIF_F_SG | NETIF_F_IP_CSUM | NETIF_F_TSO |
				NETIF_F_HIGHDMA | NETIF_F_FRAGLIST |
				NETIF_F_IPV6_CSUM;

	axdev->tx_casecade_size = TX_CASECADES_SIZE;
	axdev->gso_max_size = AX_GSO_DEFAULT_SIZE;
	axdev->mii.supports_gmii = true;
	axdev->mii.dev = netdev;
	axdev->mii.mdio_read = ax_mdio_read;
	axdev->mii.mdio_write = ax_mdio_write;
	axdev->mii.phy_id_mask = 0xff;
	axdev->mii.reg_num_mask = 0xff;
	axdev->mii.phy_id = AX88179A_PHY_ID;
	axdev->mii.force_media = false;
	axdev->mii.advertising = ADVERTISE_10HALF | ADVERTISE_10FULL |
				 ADVERTISE_100HALF | ADVERTISE_100FULL;

	netif_set_gso_max_size(netdev, axdev->gso_max_size);

	axdev->bin_setting.custom = false;
	axdev->tx_align_len = 8;

	if (ax88179a_get_mac(axdev))
		return -ENODEV;

	netdev->ethtool_ops = &ax88179a_ethtool_ops;
	axdev->netdev->netdev_ops = &ax88179a_netdev_ops;

	return ret;
}

static void ax88179a_unbind(struct ax_device *axdev)
{

}

static int ax88179a_stop(struct ax_device *axdev)
{
	u16 reg16;

	reg16 = AX_RX_CTL_STOP;
	ax_write_cmd(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE, 2, 2, &reg16);

	ax88179a_set_phy_power(axdev, false);

	return 0;
}

void ax88179a_set_multicast(struct net_device *net)
{
	struct ax_device *axdev = netdev_priv(net);
	u8 *m_filter = axdev->m_filter;
	int mc_count = 0;

	if (!test_bit(AX_ENABLE, &axdev->flags))
		return;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 35)
	mc_count = net->mc_count;
#else
	mc_count = netdev_mc_count(net);
#endif

	axdev->rxctl = (AX_RX_CTL_START | AX_RX_CTL_AB);
	if (axdev->chip_version >= 7 ||
	    axdev->ether_link.speed == ETHER_LINK_1000)
		axdev->rxctl |= AX_RX_CTL_DROPCRCERR;

	if (net->flags & IFF_PROMISC) {
		axdev->rxctl |= AX_RX_CTL_PRO;
	} else if (net->flags & IFF_ALLMULTI || mc_count > AX_MAX_MCAST) {
		axdev->rxctl |= AX_RX_CTL_AMALL;
	} else if (netdev_mc_empty(net)) {
		/* just broadcast and directed */
	} else {
		/* We use the 20 byte dev->data for our 8 byte filter buffer
		 * to avoid allocating memory that is tricky to free later
		 */
		u32 crc_bits = 0;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 35)
		struct dev_mc_list *mc_list = net->mc_list;
		int i = 0;

		memset(m_filter, 0, AX_MCAST_FILTER_SIZE);

		/* Build the multicast hash filter. */
		for (i = 0; i < net->mc_count; i++) {
			crc_bits =
			    ether_crc(ETH_ALEN,
				      mc_list->dmi_addr) >> 26;
			*(m_filter + (crc_bits >> 3)) |=
				1 << (crc_bits & 7);
			mc_list = mc_list->next;
		}
#else
		struct netdev_hw_addr *ha = NULL;
		memset(m_filter, 0, AX_MCAST_FILTER_SIZE);
		netdev_for_each_mc_addr(ha, net) {
			crc_bits = ether_crc(ETH_ALEN, ha->addr) >> 26;
			*(m_filter + (crc_bits >> 3)) |= 1 << (crc_bits & 7);
		}
#endif
		ax_write_cmd_async(axdev, AX_ACCESS_MAC, AX_MULTI_FILTER_ARRY,
				   AX_MCAST_FILTER_SIZE, AX_MCAST_FILTER_SIZE,
				   m_filter);

		axdev->rxctl |= AX_RX_CTL_AM;
	}

	ax_write_cmd_async(axdev, AX_ACCESS_MAC, AX_RX_CTL,
			   2, 2, &axdev->rxctl);
}

static int ax88179a_hw_init(struct ax_device *axdev)
{
	u16 reg16;
	u8 reg8;
	int ret;

	ret = ax88179a_set_phy_power(axdev, true);
	if (ret < 0)
		return ret;
	msleep(250);

	reg8 = AX_TXCOE_IP | AX_TXCOE_TCP | AX_TXCOE_UDP |
	       AX_TXCOE_TCPV6 | AX_TXCOE_UDPV6;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_TXCOE_CTL, 1, 1, &reg8);
	if (ret < 0)
		return ret;

	reg8 = AX_RXCOE_IP | AX_RXCOE_TCP | AX_RXCOE_UDP |
	       AX_RXCOE_TCPV6 | AX_RXCOE_UDPV6;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_RXCOE_CTL, 1, 1, &reg8);
	if (ret < 0)
		return ret;

	reg8 = AX_MAC_EFF_EN;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_MAC_BULK_OUT_CTRL,
			    1, 1, &reg8);
	if (ret < 0)
		return ret;

	reg16 = 0;
	ret = ax_write_cmd_async(axdev, AX_ACCESS_MAC, AX_RX_CTL, 2, 2, &reg16);
	if (ret < 0)
		return ret;

	reg8 = 0x04;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_PAUSE_WATERLVL_LOW,
			    1, 1, &reg8);
	if (ret < 0)
		return ret;

	reg8 = 0x10;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_PAUSE_WATERLVL_HIGH,
			    1, 1, &reg8);
	if (ret < 0)
		return ret;

	reg8 = 0;
	if (axdev->netdev->features & NETIF_F_HW_VLAN_CTAG_FILTER)
		reg8 |= AX_VLAN_CONTROL_VFE;
	if (axdev->netdev->features & NETIF_F_HW_VLAN_CTAG_RX)
		reg8 |= AX_VLAN_CONTROL_VSO;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_VLAN_ID_CONTROL,
			   1, 1, &reg8);
	if (ret < 0)
		return ret;

	reg8 = 0xff;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_MAC_BM_INT_MASK, 
			    1, 1, &reg8);
	if (ret < 0)
		return ret;

	reg8 = 0;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_MAC_BM_RX_DMA_CTL,
			    1, 1, &reg8);
	if (ret < 0)
		return ret;


	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_MAC_BM_TX_DMA_CTL, 
			    1, 1, &reg8);
	if (ret < 0)
		return ret;


	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_MAC_ARC_CTRL, 
			    1, 1, &reg8);
	if (ret < 0)
		return ret;


	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_MAC_SWP_CTRL, 
			    1, 1, &reg8);
	if (ret < 0)
		return ret;


	reg8 = AX_TXHDR_CKSUM_EN;		
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_MAC_TX_HDR_CKSUM, 
				1, 1, &reg8);
	if (ret < 0)
		return ret;

	reg16 = AX_RX_CTL_START | AX_RX_CTL_AP | AX_RX_CTL_AMALL |
		AX_RX_CTL_AB;
	if (axdev->chip_version >= 7 ||
	    axdev->ether_link.speed == ETHER_LINK_1000)
		reg16 |= AX_RX_CTL_DROPCRCERR;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_RX_CTL, 1, 1, &reg16);
	if (ret < 0)
		return ret;

	reg8 = 0;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_MAC_PATH, 1, 1, &reg8);
	if (ret < 0)
		return ret;

	ret = ax_read_cmd(axdev, AX_ACCESS_MAC, AX_MONITOR_MODE, 1, 1, &reg8, 1);
	if (ret < 0)
		return ret;
	reg8 &= 0xE0;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_MONITOR_MODE, 1, 1, &reg8);
	if (ret < 0)
		return ret;


	ret = ax_read_cmd(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
				 2, 2, &reg16, 2);
	if (ret < 0)
		return ret;

	reg16 &= ~AX_MEDIUM_GIGAMODE;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
			    2, 2, &reg16);
	if (ret < 0)
		return ret;

	reg16 |= AX_MEDIUM_GIGAMODE;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
			    2, 2, &reg16);
	if (ret < 0)
		return ret;

	reg8 = 0;
	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_BFM_DATA, 1, 1, &reg8);
	if (ret < 0)
		return ret;

	ret = ax_write_cmd(axdev, AX_ACCESS_MAC, AX88179A_CDC_ECM_CTRL,
			   1, 1, &reg8);
	if (ret < 0)
		return ret;

	ax88179a_Gether_setting(axdev);
	ax_set_tx_qlen(axdev);

	return ret;
}

static int ax88179a_get_ether_link(struct ax_device *axdev)
{
	struct ax_ether_link *ether_link = &axdev->ether_link;
	u16 phy_1G, phy_100M, phy_10M;
	u16 phy_anar, phy_anlpar;
	int ret = 0;

	if (axdev->int_link_speed) {
		ether_link->speed = axdev->int_link_info & 0xF;
		ether_link->full_duplex =
			(axdev->int_link_info & 0x10)?true:false;
	}

	if (ether_link->speed > ETHER_LINK_NONE)
		return 0;

	ret = ax_read_cmd_nopm(axdev, AX_ACCESS_PHY, AX88179A_PHY_ID,
			       PHY_1000M_STS, 2, &phy_1G, 1);
	if (ret < 0) 
		return ret;

	ret = ax_read_cmd_nopm(axdev, AX_ACCESS_PHY, AX88179A_PHY_ID,
			       PHY_100M_STS, 2, &phy_100M, 1);
	if (ret < 0)
		return ret;

	ret = ax_read_cmd_nopm(axdev, AX_ACCESS_PHY, AX88179A_PHY_ID,
			       PHY_10M_STS, 2, &phy_10M, 1);
	if (ret < 0)
		return ret;

	ret = ax_read_cmd_nopm(axdev, AX_ACCESS_PHY, AX88179A_PHY_ID,
			       GMII_PHY_ANAR, 2, &phy_anar, 1);
	if (ret < 0)
		return ret;

	ret = ax_read_cmd_nopm(axdev, AX_ACCESS_PHY, AX88179A_PHY_ID,
			       GMII_PHY_ANLPAR, 2, &phy_anlpar, 1);
	if (ret < 0)
		return ret;

	if (phy_1G & LINK_1000M_OK) {
		ether_link->speed = ETHER_LINK_1000;
		ether_link->full_duplex = true;
	} else if (phy_100M & LINK_100M_OK){
		ether_link->speed = ETHER_LINK_100;

		if ((phy_anar & GMII_ANAR_100TXFD) && 
		    (phy_anlpar & GMII_ANLPAR_100TXFD))
			ether_link->full_duplex = true;
		else
			ether_link->full_duplex = false;
	} else if (phy_10M & LINK_10M_OK) {
		ether_link->speed = ETHER_LINK_10;

		if ((phy_anar & GMII_ANAR_10TFD) && 
		    (phy_anlpar & GMII_ANLPAR_10TFD))
			ether_link->full_duplex = true;
		else
			ether_link->full_duplex = false;
	} else {
		ether_link->speed = ETHER_LINK_NONE;
		ether_link->full_duplex = false;
		return -EINVAL;
	}

	return 0;
}

static int ax88179a_set_bulkin_setting(struct ax_device *axdev)
{
	struct ax_ether_link *ether_link = &axdev->ether_link;
	u8 link_sts;
	int index, ret;

	ret = ax_read_cmd_nopm(axdev, AX_ACCESS_MAC, PHYSICAL_LINK_STATUS,
			       1, 1, &link_sts, 0);
	if (ret < 0)
		return ret;

	switch (ether_link->speed) {
	case ETHER_LINK_1000:
		if (link_sts & AX_USB_SS)
			index = 0;
		else if (link_sts & AX_USB_HS)
			index = 1;
		break;
	case ETHER_LINK_100:
		if (link_sts & AX_USB_SS)
			index = 2;
		else if (link_sts & AX_USB_HS)
			index = 4;

		if (!ether_link->full_duplex)
			index++;
		break;
	case ETHER_LINK_10:
		index = 6;
		break;
	default:
		index = 0;
		break;
	};

	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_RX_BULKIN_QCTRL,
				5, 5, &AX88179A_BULKIN_SIZE[index]);
	if (ret < 0)
		return ret;

	return 0;
}

static int ax88179a_link_setting(struct ax_device *axdev)
{
	struct ax_ether_link *ether_link = &axdev->ether_link;
	u16 medium_mode, rxctl, reg16;
	u8 reg8[3];
	int ret;

	rxctl = AX_RX_CTL_STOP;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_RX_CTL, 2, 2, &rxctl);
	if (ret < 0)
		return ret;

	reg16 = 0;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_MAC_PATH,
				1, 1, &reg16);
	if (ret < 0)
		return ret;

	reg8[0] = 0xA5;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_MAC_CDC_DELAY_TX,
				 1, 1, reg8);
	if (ret < 0)
		return ret;

	reg8[0] = 0x10;
	reg8[1] = 0x04;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_PAUSE_WATERLVL_HIGH,
				2, 2, reg8);
	if (ret < 0)
		return ret;

	rxctl = AX_RX_CTL_START | AX_RX_CTL_AP | AX_RX_CTL_AMALL |
		AX_RX_CTL_AB;
	if (ether_link->speed == ETHER_LINK_1000)
		rxctl |= AX_RX_CTL_DROPCRCERR;

	medium_mode = AX_MEDIUM_RECEIVE_EN | AX_MEDIUM_TXFLOW_CTRLEN |
		      AX_MEDIUM_RXFLOW_CTRLEN;
	switch (ether_link->speed) {
	case ETHER_LINK_1000:
	case ETHER_LINK_100:
		reg8[0] = 0x78;
		reg8[1] = (AX_LSOFC_WCNT_7_ACCESS << 5) | AX_GMII_CRC_APPEND;
		reg8[2] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88178A_MAC_RX_STATUS_CDC, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0x40;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_RX_DATA_CDC_CNT,
					1, 1, reg8);
		if (ret < 0)
			return ret;

		medium_mode |= AX_MEDIUM_GIGAMODE;
		break;
	case ETHER_LINK_10:
		reg8[0] = 0xFA;
		reg8[1] = (AX_LSOFC_WCNT_7_ACCESS << 5) | AX_GMII_CRC_APPEND;
		reg8[2] = 0xFF;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88178A_MAC_RX_STATUS_CDC, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0xFA;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_RX_DATA_CDC_CNT,
					1, 1, reg8);
		if (ret < 0)
			return ret;
		break;
	default:
		break;
	};

	reg8[0] = 0;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_BFM_DATA,
				1, 1, reg8);
	if (ret < 0)
		return ret;

	ret = ax88179a_set_bulkin_setting(axdev);
	if (ret < 0)
		return ret;

	if (ether_link->full_duplex) medium_mode |= AX_MEDIUM_FULL_DUPLEX;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
				2, 2, &medium_mode);
	if (ret < 0)
		return ret;

	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_RX_CTL, 2, 2, &rxctl);
	if (ret < 0)
		return ret;

	reg16 = AX_MAC_RX_PATH_READY | AX_MAC_TX_PATH_READY;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_MAC_PATH,
				1, 1, &reg16);
	if (ret < 0)
		return ret;

	return 0;
}

static int ax88179a_link_reset(struct ax_device *axdev)
{
	int ret;

	ret = ax88179a_get_ether_link(axdev);
	if (ret < 0)
		return ret;

	ret = ax88179a_link_setting(axdev);
	if (ret < 0)
		return ret;

	return 0;
}

static int ax88279_set_bulkin_setting(struct ax_device *axdev)
{
	struct ax_ether_link *ether_link = &axdev->ether_link;
	u8 link_sts;
	int index, ret;

	ret = ax_read_cmd_nopm(axdev, AX_ACCESS_MAC, PHYSICAL_LINK_STATUS,
			       1, 1, &link_sts, 0);
	if (ret < 0)
		return ret;

	if (link_sts & AX_USB_FS) {
		index = 7;
	} else {
		switch (ether_link->speed) {
		case ETHER_LINK_2500:
			index = 0;
			break;
		case ETHER_LINK_1000:
			if (link_sts & AX_USB_SS)
				index = 1;
			else if (link_sts & AX_USB_HS)
				index = 2;
			break;
		case ETHER_LINK_100:
			if (link_sts & AX_USB_SS)
				index = 3;
			else if (link_sts & AX_USB_HS)
				index = 5;

			if (!ether_link->full_duplex)
				index++;
			break;
		case ETHER_LINK_10:
			break;
		default:
			return -1;
		};
	}

	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_RX_BULKIN_QCTRL,
				5, 5, &AX88279_BULKIN_SIZE[index]);
	if (ret < 0) return ret;

	return 0;
}

static int ax88279_link_setting(struct ax_device *axdev)
{
	struct ax_ether_link *ether_link = &axdev->ether_link;
	u16 medium_mode, rxctl, reg16;
	u8 reg8[3];
	int ret;

	rxctl = AX_RX_CTL_STOP;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_RX_CTL, 2, 2, &rxctl);
	if (ret < 0)
		return ret;

	reg16 = 0;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_MAC_PATH,
				1, 1, &reg16);
	if (ret < 0)
		return ret;

	reg8[0] = 0xA5;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_MAC_CDC_DELAY_TX,
				 1, 1, reg8);
	if (ret < 0)
		return ret;

	reg8[0] = 0x10;
	reg8[1] = 0x04;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
				AX_PAUSE_WATERLVL_HIGH, 2, 2, reg8);
	if (ret < 0)
		return ret;

	rxctl = AX_RX_CTL_START | AX_RX_CTL_AP | AX_RX_CTL_AMALL |
		AX_RX_CTL_AB | AX_RX_CTL_DROPCRCERR;

	reg8[0] = 0;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
				AX88179A_ETH_TX_GAP, 1, 1, reg8);
	if (ret < 0)
		return ret;

	/* TODO:: */
	reg8[0] = 0x07;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, 0xF9, 1, 1,
				&reg8);
	if (ret < 0)
		return ret;

	medium_mode = AX_MEDIUM_RECEIVE_EN | AX_MEDIUM_TXFLOW_CTRLEN |
		      AX_MEDIUM_RXFLOW_CTRLEN;	
	switch (ether_link->speed) {
	case ETHER_LINK_2500:
		reg8[0] = 0x00;
		reg8[1] = 0xF8;
		reg8[2] = 0x07;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_TX_PAUSE_0, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0x78;
		reg8[1] = (AX_LSOFC_WCNT_7_ACCESS << 5);
		reg8[2] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88178A_MAC_RX_STATUS_CDC, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0x40;
		reg8[1] = AX_MAC_MIQFFCTRL_FORMAT | AX_MAC_MIQFFCTRL_DROP_CRC |
			  AX_MAC_LSO_ERR_EN;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_RX_DATA_CDC_CNT,
					2, 2, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = AX_XGMII_EN;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					 AX88179A_BFM_DATA, 1, 1, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = (0x1C) | AX_LSO_ENHANCE_EN;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_LSO_ENHANCE_CTRL, 1, 1,
					&reg8);
		if (ret < 0)
			return ret;

		medium_mode |= AX_MEDIUM_GIGAMODE;
		break;
	case ETHER_LINK_1000:
		reg8[0] = 0x48;
		reg8[1] = 0xF1;
		reg8[2] = 0x3E;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					 AX88179A_MAC_TX_PAUSE_0, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0x78;
		reg8[1] = (AX_LSOFC_WCNT_7_ACCESS << 5);
		reg8[2] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88178A_MAC_RX_STATUS_CDC, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0x40;
		reg8[1] = AX_MAC_MIQFFCTRL_FORMAT | AX_MAC_MIQFFCTRL_DROP_CRC |
			  AX_MAC_LSO_ERR_EN;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_RX_DATA_CDC_CNT,
					2, 2, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					 AX88179A_BFM_DATA, 1, 1, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_LSO_ENHANCE_CTRL, 1, 1,
					&reg8);
		if (ret < 0)
			return ret;

		medium_mode |= AX_MEDIUM_GIGAMODE;
		break;
	case ETHER_LINK_100:
		reg8[0] = 0x90;
		reg8[1] = 0xE2;
		reg8[2] = 0x7D;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					 AX88179A_MAC_TX_PAUSE_0, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0x78;
		reg8[1] = (AX_LSOFC_WCNT_7_ACCESS << 5);
		reg8[2] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88178A_MAC_RX_STATUS_CDC, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0x40;
		reg8[1] = AX_MAC_MIQFFCTRL_FORMAT | AX_MAC_MIQFFCTRL_DROP_CRC |
			  AX_MAC_LSO_ERR_EN;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_RX_DATA_CDC_CNT,
					2, 2, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_BFM_DATA, 1, 1, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_LSO_ENHANCE_CTRL, 1, 1,
					&reg8);
		if (ret < 0)
			return ret;
		break;
	case ETHER_LINK_10:
		reg8[0] = 0x90;
		reg8[1] = 0xE2;
		reg8[2] = 0x7D;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_TX_PAUSE_0, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0xFA;
		reg8[1] = (AX_LSOFC_WCNT_7_ACCESS << 5);
		reg8[2] = 0xFF;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88178A_MAC_RX_STATUS_CDC, 3, 3, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0xFA;
		reg8[1] = AX_MAC_MIQFFCTRL_FORMAT | AX_MAC_MIQFFCTRL_DROP_CRC |
			  AX_MAC_LSO_ERR_EN;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_RX_DATA_CDC_CNT,
					2, 2, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_BFM_DATA, 1, 1, reg8);
		if (ret < 0)
			return ret;

		reg8[0] = 0;
		ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC,
					AX88179A_MAC_LSO_ENHANCE_CTRL, 1, 1,
					&reg8);
		if (ret < 0)
			return ret;
		break;
	default:
		break;
	};

	ret = ax88279_set_bulkin_setting(axdev);
	if (ret < 0)
		return ret;

	if (ether_link->full_duplex)
		medium_mode |= AX_MEDIUM_FULL_DUPLEX;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_MEDIUM_STATUS_MODE,
				2, 2, &medium_mode);
	if (ret < 0)
		return ret;

	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_RX_CTL, 2, 2, &rxctl);
	if (ret < 0)
		return ret;

	reg16 = AX_MAC_RX_PATH_READY | AX_MAC_TX_PATH_READY;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_MAC_PATH,
				1, 1, &reg16);
	if (ret < 0)
		return ret;

	return 0;
}

static int ax88279_link_reset(struct ax_device *axdev)
{
	int ret;

	ret = ax88179a_get_ether_link(axdev);
	if (ret < 0)
		return ret;

	ret = ax88279_link_setting(axdev);
	if (ret < 0)
		return ret;

	return 0;
}

inline void ax88179a_rx_checksum(struct sk_buff *skb, void *pkt_hdr)
{
	struct _179a_rx_pkt_header *hdr = (struct _179a_rx_pkt_header *)pkt_hdr;
	skb->ip_summed = CHECKSUM_NONE;

	if ((hdr->L4_err) ||
	    (hdr->L3_err))
		return;

	if ((hdr->L4_pkt_type == AX_RXHDR_L4_TYPE_TCP) ||
	    (hdr->L4_pkt_type == AX_RXHDR_L4_TYPE_UDP))
		skb->ip_summed = CHECKSUM_UNNECESSARY;
}

static int ax88179a_rx_fixup(struct ax_device *axdev, struct rx_desc *desc,
			     int done_already, int budget)
{
	struct napi_struct *napi = &axdev->napi;
	struct net_device *netdev = axdev->netdev;
	struct net_device_stats *stats = ax_get_stats(netdev);
	struct _179a_rx_pkt_header *pkt_hdr;
	struct _179a_rx_header *rx_header;
	const u32 actual_length = desc->urb->actual_length;
	int work_done = done_already;
	u8 *rx_data;
	u16 pkt_count = 0;

	rx_header = (struct _179a_rx_header *)
			(desc->head + actual_length - AX88179A_RX_HEADER_SIZE);
	le64_to_cpus(rx_header);

	pkt_count = rx_header->pkt_cnt;
	if (((actual_length - ((pkt_count + 1) * 8)) != rx_header->hdr_off) ||
	    rx_header->hdr_off >= actual_length ||
	    pkt_count == 0) {
		desc->urb->actual_length = 0;
		stats->rx_length_errors++;
		return work_done;
	}	

	pkt_hdr = (struct _179a_rx_pkt_header *)
			(desc->head + rx_header->hdr_off);
	
	rx_data = desc->head;
	while (pkt_count--) {
		struct sk_buff *skb;
		u32 pkt_len;

		le64_to_cpus(pkt_hdr);

		if (!pkt_hdr->RxOk) {
			stats->rx_crc_errors++;
			goto find_next_rx;
		}

		pkt_len = (u32)(pkt_hdr->length & 0x7FFF);

		if (pkt_hdr->drop) {
			stats->rx_dropped++;
			goto find_next_rx;
		}

		skb = napi_alloc_skb(napi, pkt_len);
		if (!skb) {
			stats->rx_dropped++;
			goto find_next_rx;
		}

		skb_put(skb, pkt_len);
		memcpy(skb->data, rx_data, pkt_len);
		ax88179a_rx_checksum(skb, pkt_hdr);

		skb->truesize = skb->len + sizeof(struct sk_buff);

		if (pkt_hdr->vlan_ind) {
			__vlan_hwaccel_put_tag(skb, htons(ETH_P_8021Q),
					     pkt_hdr->vlan_tag & VLAN_VID_MASK);
		}

		skb->protocol = eth_type_trans(skb, netdev);
		if (work_done < budget) {
			napi_gro_receive(napi, skb);
			work_done++;
			stats->rx_packets++;
			stats->rx_bytes += pkt_len;
		} else {
			__skb_queue_tail(&axdev->rx_queue, skb);
		}
find_next_rx:
		rx_data += (pkt_len + 7) & 0x7FFF8;
		pkt_hdr++;
	}

	return work_done;
}

static int ax88179a_tx_fixup(struct ax_device *axdev, struct tx_desc *desc)
{
	struct sk_buff_head skb_head, *tx_queue = &axdev->tx_queue;
	struct net_device_stats *stats = &axdev->netdev->stats;
	int remain, ret;
	u8 *tx_data;

	__skb_queue_head_init(&skb_head);
	spin_lock(&tx_queue->lock);
	skb_queue_splice_init(tx_queue, &skb_head);
	spin_unlock(&tx_queue->lock);

	tx_data = desc->head;
	desc->skb_num = 0;
	desc->skb_len = 0;
	remain = axdev->tx_casecade_size;

	while (remain >= (ETH_ZLEN + AX88179A_TX_HEADER_SIZE)) {
		struct sk_buff *skb;
		struct _179a_tx_pkt_header *tx_hdr;
		u16 tci = 0;

		skb = __skb_dequeue(&skb_head);
		if (!skb)
			break;

		if ((skb->len + AX88179A_TX_HEADER_SIZE) > remain) {
			__skb_queue_head(&skb_head, skb);
			break;
		}

		tx_hdr = (struct _179a_tx_pkt_header *)tx_data;
		memset(tx_hdr, 0, AX88179A_TX_HEADER_SIZE);
		tx_hdr->length = (skb->len & 0x1FFFFF);
		tx_hdr->checksum = AX88179A_TX_HERDER_CHKSUM(tx_hdr->length);
		tx_hdr->max_seg_size = skb_shinfo(skb)->gso_size;
		if ((axdev->netdev->features & NETIF_F_HW_VLAN_CTAG_TX) && 
		    (vlan_get_tag(skb, &tci) >= 0)) {
			tx_hdr->vlan_tag = 1;
			tx_hdr->vlan_info = tci;
		}

		cpu_to_le64s(tx_hdr);
		tx_data += AX88179A_TX_HEADER_SIZE;

		if (skb_copy_bits(skb, 0, tx_data, skb->len) < 0) {
			stats->tx_dropped++;
			dev_kfree_skb_any(skb);
			continue;
		}

		tx_data = __tx_buf_align((void *)(tx_data + skb->len),
					 axdev->tx_align_len);
		desc->skb_len += skb->len;
		desc->skb_num++;
		dev_kfree_skb_any(skb);

		if (tx_hdr->max_seg_size)
			break;

		remain = axdev->tx_casecade_size - 
			 (int)((void *)tx_data - desc->head);
	}

	if (!skb_queue_empty(&skb_head)) {
		spin_lock(&tx_queue->lock);
		skb_queue_splice(&skb_head, tx_queue);
		spin_unlock(&tx_queue->lock);
	}

	netif_tx_lock(axdev->netdev);
	if (netif_queue_stopped(axdev->netdev) &&
	    skb_queue_len(tx_queue) < axdev->tx_qlen) {
		netif_wake_queue(axdev->netdev);
	}
	netif_tx_unlock(axdev->netdev);

	ret = usb_autopm_get_interface_async(axdev->intf);
	if (ret < 0)
		return ret;

	usb_fill_bulk_urb(desc->urb, axdev->udev,
			  usb_sndbulkpipe(axdev->udev, 3),
			  desc->head, (int)(tx_data - (u8 *)desc->head),
			  (usb_complete_t)ax_write_bulk_callback, desc);

	ret = usb_submit_urb(desc->urb, GFP_ATOMIC);
	if (ret < 0)
		usb_autopm_put_interface_async(axdev->intf);

	return 0;
}

static int ax88179a_system_suspend(struct ax_device *axdev)
{
	u16 reg16;
	int ret;

	reg16 = 0;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX88179A_BFM_DATA,
				2, 2, &reg16);
	if (ret < 0)
		return ret;

	reg16 = AX_RX_CTL_STOP;
	ret = ax_write_cmd_nopm(axdev, AX_ACCESS_MAC, AX_RX_CTL, 2, 2, &reg16);
	if (ret < 0)
		return ret;

	return 0;
}

static int ax88179a_system_resume(struct ax_device *axdev)
{
	int ret;

	if (!ax88179a_check_phy_power(axdev))
		ax88179a_set_phy_power(axdev, true);

	ret = ax_write_cmd_nopm(axdev, AX_FW_MODE, AX_FW_MODE_179A, 0, 0, NULL);
	if (ret < 0)
		return ret;

	axdev->driver_info->hw_init(axdev);

	return 0;
}

const struct driver_info ax88279_info = {
	.bind		= ax88179a_bind,
	.unbind		= ax88179a_unbind,
	.hw_init	= ax88179a_hw_init,
	.stop		= ax88179a_stop,
	.link_reset	= ax88279_link_reset,
	.rx_fixup	= ax88179a_rx_fixup,
	.tx_fixup	= ax88179a_tx_fixup,
	.system_suspend = ax88179a_system_suspend,
	.system_resume	= ax88179a_system_resume,
	.napi_weight	= AX88179A_NAPI_WEIGHT,
	.buf_rx_size	= AX88179A_BUF_RX_SIZE,
	.chip_mode	= AX_CHIP_AX88279,
};

const struct driver_info ax88179a_info = {
	.bind		= ax88179a_bind,
	.unbind		= ax88179a_unbind,
	.hw_init	= ax88179a_hw_init,
	.stop		= ax88179a_stop,
	.link_reset	= ax88179a_link_reset,
	.rx_fixup	= ax88179a_rx_fixup,
	.tx_fixup	= ax88179a_tx_fixup,
	.system_suspend = ax88179a_system_suspend,
	.system_resume	= ax88179a_system_resume,
	.napi_weight	= AX88179A_NAPI_WEIGHT,
	.buf_rx_size	= AX88179A_BUF_RX_SIZE,
	.chip_mode	= AX_CHIP_AX88179A,
};

const struct driver_info ax88772d_info = {
	.bind		= ax88179a_bind,
	.unbind		= ax88179a_unbind,
	.hw_init	= ax88179a_hw_init,
	.stop		= ax88179a_stop,
	.link_reset	= ax88179a_link_reset,
	.rx_fixup	= ax88179a_rx_fixup,
	.tx_fixup	= ax88179a_tx_fixup,
	.system_suspend = ax88179a_system_suspend,
	.system_resume	= ax88179a_system_resume,
	.napi_weight	= AX88179A_NAPI_WEIGHT,
	.buf_rx_size	= AX88179A_BUF_RX_SIZE,
	.chip_mode	= AX_CHIP_AX88772D,
};
